/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage.ldap.idm.store.ldap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.naming.AuthenticationException;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.AttributeInUseException;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.NoSuchAttributeException;
import javax.naming.directory.SchemaViolationException;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.ModelException;
import org.keycloak.representations.idm.LDAPCapabilityRepresentation;
import org.keycloak.storage.ldap.LDAPConfig;
import org.keycloak.storage.ldap.idm.model.LDAPDn;
import org.keycloak.storage.ldap.idm.model.LDAPObject;
import org.keycloak.storage.ldap.idm.query.Condition;
import org.keycloak.storage.ldap.idm.query.internal.EqualCondition;
import org.keycloak.storage.ldap.idm.query.internal.LDAPQuery;
import org.keycloak.storage.ldap.idm.query.internal.LDAPQueryConditionsBuilder;
import org.keycloak.storage.ldap.idm.store.IdentityStore;
import org.keycloak.storage.ldap.idm.store.ldap.LDAPOperationManager;
import org.keycloak.storage.ldap.mappers.LDAPOperationDecorator;

public class LDAPIdentityStore
implements IdentityStore {
    private static final Logger logger = Logger.getLogger(LDAPIdentityStore.class);
    private static final Pattern rangePattern = Pattern.compile("([^;]+);range=([0-9]+)-([0-9]+|\\*)");
    private final LDAPConfig config;
    private final LDAPOperationManager operationManager;

    public LDAPIdentityStore(KeycloakSession session, LDAPConfig config) {
        this.config = config;
        this.operationManager = new LDAPOperationManager(session, config);
    }

    @Override
    public LDAPConfig getConfig() {
        return this.config;
    }

    @Override
    public void add(LDAPObject ldapObject) {
        if (ldapObject.getUuid() != null) {
            throw new ModelException("Can't add object with already assigned uuid");
        }
        BasicAttributes ldapAttributes = this.extractAttributesForSaving(ldapObject, true);
        ldapObject.setUuid(this.operationManager.createSubContext(ldapObject.getDn().getLdapName(), ldapAttributes));
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and dn [%s] successfully added to LDAP store.", (Object)ldapObject.getUuid(), (Object)ldapObject.getDn());
        }
    }

    @Override
    public void addMemberToGroup(LdapName groupDn, String memberAttrName, String value) {
        BasicAttribute attr = new BasicAttribute(memberAttrName, value);
        ModificationItem item = new ModificationItem(1, attr);
        try {
            this.operationManager.modifyAttributesNaming(groupDn, new ModificationItem[]{item}, null);
        }
        catch (NameAlreadyBoundException | AttributeInUseException e) {
            logger.debugf("Group %s already contains the member %s", (Object)groupDn, (Object)value);
        }
        catch (NamingException e) {
            throw new ModelException("Could not modify attribute for DN [" + groupDn + "]", (Throwable)e);
        }
    }

    @Override
    public void removeMemberFromGroup(LdapName groupDn, String memberAttrName, String value) {
        BasicAttribute attr = new BasicAttribute(memberAttrName, value);
        ModificationItem item = new ModificationItem(3, attr);
        try {
            this.operationManager.modifyAttributesNaming(groupDn, new ModificationItem[]{item}, null);
        }
        catch (NoSuchAttributeException e) {
            logger.debugf("Group %s does not contain the member %s", (Object)groupDn, (Object)value);
        }
        catch (SchemaViolationException e) {
            logger.infof("Schema violation in group %s removing member %s. Trying adding empty member attribute.", (Object)groupDn, (Object)value);
            try {
                this.operationManager.modifyAttributesNaming(groupDn, new ModificationItem[]{item, new ModificationItem(1, new BasicAttribute(memberAttrName, "cn=empty-membership-placeholder"))}, null);
            }
            catch (NamingException ex) {
                throw new ModelException("Could not modify attribute for DN [" + groupDn + "]", (Throwable)ex);
            }
        }
        catch (NamingException e) {
            throw new ModelException("Could not modify attribute for DN [" + groupDn + "]", (Throwable)e);
        }
    }

    @Override
    public void update(LDAPObject ldapObject) {
        this.checkRename(ldapObject);
        BasicAttributes updatedAttributes = this.extractAttributesForSaving(ldapObject, false);
        NamingEnumeration<Attribute> attributes = updatedAttributes.getAll();
        this.operationManager.modifyAttributes(ldapObject.getDn().getLdapName(), attributes);
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and DN [%s] successfully updated to LDAP store.", (Object)ldapObject.getUuid(), (Object)ldapObject.getDn());
        }
    }

    protected void checkRename(LDAPObject ldapObject) {
        String rdnAttrVal;
        LDAPDn.RDN firstRdn = ldapObject.getDn().getFirstRdn();
        LdapName oldDn = ldapObject.getDn().getLdapName();
        List<String> toUpdateKeys = firstRdn.getAllKeys();
        toUpdateKeys.retainAll(ldapObject.getRdnAttributeNames());
        List<String> toRemoveKeys = firstRdn.getAllKeys();
        toRemoveKeys.removeAll(ldapObject.getRdnAttributeNames());
        ArrayList<String> toAddKeys = new ArrayList<String>(ldapObject.getRdnAttributeNames());
        toAddKeys.removeAll(firstRdn.getAllKeys());
        boolean changed = false;
        for (String attrKey : toUpdateKeys) {
            String oldRdnAttrVal;
            if (ldapObject.getReadOnlyAttributeNames().contains(attrKey.toLowerCase()) || (rdnAttrVal = ldapObject.getAttributeAsString(attrKey)) == null || (oldRdnAttrVal = firstRdn.getAttrValue(attrKey)).equalsIgnoreCase(rdnAttrVal)) continue;
            changed = true;
            firstRdn.setAttrValue(attrKey, rdnAttrVal);
        }
        for (String attrKey : toAddKeys) {
            rdnAttrVal = ldapObject.getAttributeAsString(attrKey);
            if (rdnAttrVal == null) continue;
            changed = true;
            firstRdn.setAttrValue(attrKey, rdnAttrVal);
        }
        for (String attrKey : toRemoveKeys) {
            changed |= firstRdn.removeAttrValue(attrKey);
        }
        if (changed) {
            LDAPDn newLdapDn = ldapObject.getDn().getParentDn();
            newLdapDn.addFirst(firstRdn);
            LdapName newDn = newLdapDn.getLdapName();
            logger.debugf("Renaming LDAP Object. Old DN: [%s], New DN: [%s]", (Object)oldDn, (Object)newDn);
            newDn = this.operationManager.renameEntry(oldDn, newDn, true);
            ldapObject.setDn(LDAPDn.fromLdapName(newDn));
        }
    }

    @Override
    public void remove(LDAPObject ldapObject) {
        this.operationManager.removeEntry(ldapObject.getDn().getLdapName());
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and DN [%s] successfully removed from LDAP store.", (Object)ldapObject.getUuid(), (Object)ldapObject.getDn().toString());
        }
    }

    @Override
    public List<LDAPObject> fetchQueryResults(LDAPQuery identityQuery) {
        if (identityQuery.getSorting() != null && !identityQuery.getSorting().isEmpty()) {
            throw new ModelException("LDAP Identity Store does not yet support sorted queries.");
        }
        ArrayList<LDAPObject> results = new ArrayList<LDAPObject>();
        try {
            LdapName baseDN = identityQuery.getSearchDn();
            for (Condition condition : identityQuery.getConditions()) {
                EqualCondition equalCondition;
                String uuidAttrName = this.getConfig().getUuidLDAPAttributeName();
                if (!(condition instanceof EqualCondition) || !(equalCondition = (EqualCondition)condition).getParameterName().equalsIgnoreCase(uuidAttrName)) continue;
                SearchResult search = this.operationManager.lookupById(baseDN, equalCondition.getValue().toString(), identityQuery.getReturningLdapAttributes());
                if (search != null) {
                    results.add(this.populateAttributedType(search, identityQuery));
                }
                return results;
            }
            Condition condition = this.createIdentityTypeSearchFilter(identityQuery);
            List<SearchResult> search = this.getConfig().isPagination() && identityQuery.getLimit() > 0 ? this.operationManager.searchPaginated(baseDN, condition, identityQuery) : this.operationManager.search(baseDN, condition, identityQuery.getReturningLdapAttributes(), identityQuery.getSearchScope());
            for (SearchResult result : search) {
                if (identityQuery.getSearchScope() == 2 && baseDN.equals(new LdapName(result.getNameInNamespace()))) continue;
                results.add(this.populateAttributedType(result, identityQuery));
            }
        }
        catch (NameNotFoundException e) {
            if (identityQuery.getSearchScope() == 0) {
                return Collections.emptyList();
            }
            throw new ModelException("Querying of LDAP failed " + identityQuery, (Throwable)e);
        }
        catch (Exception e) {
            throw new ModelException("Querying of LDAP failed " + identityQuery, (Throwable)e);
        }
        return results;
    }

    @Override
    public int countQueryResults(LDAPQuery identityQuery) {
        int limit = identityQuery.getLimit();
        identityQuery.setLimit(0);
        int resultCount = identityQuery.getResultList().size();
        identityQuery.setLimit(limit);
        return resultCount;
    }

    @Override
    public Set<LDAPCapabilityRepresentation> queryServerCapabilities() {
        LinkedHashSet<LDAPCapabilityRepresentation> result = new LinkedHashSet<LDAPCapabilityRepresentation>();
        try {
            ArrayList<String> attrs = new ArrayList<String>();
            attrs.add("supportedControl");
            attrs.add("supportedExtension");
            attrs.add("supportedFeatures");
            List<SearchResult> searchResults = this.operationManager.search(new LdapName(Collections.emptyList()), new LDAPQueryConditionsBuilder().present("objectclass"), Collections.unmodifiableCollection(attrs), 0);
            if (searchResults.size() != 1) {
                throw new ModelException("Could not query root DSE: unexpected result size");
            }
            SearchResult rootDse = searchResults.get(0);
            Attributes attributes = rootDse.getAttributes();
            for (String attr : attrs) {
                Attribute attribute = attributes.get(attr);
                if (null == attribute) continue;
                LDAPCapabilityRepresentation.CapabilityType capabilityType = LDAPCapabilityRepresentation.CapabilityType.fromRootDseAttributeName((String)attr);
                NamingEnumeration<?> values = attribute.getAll();
                while (values.hasMoreElements()) {
                    Object o = values.nextElement();
                    LDAPCapabilityRepresentation capability = new LDAPCapabilityRepresentation(o, capabilityType);
                    logger.info((Object)("rootDSE query: " + capability));
                    result.add(capability);
                }
            }
            return result;
        }
        catch (NamingException e) {
            throw new ModelException("Failed to query root DSE: " + e.getMessage(), (Throwable)e);
        }
    }

    @Override
    public void validatePassword(LDAPObject user, String password) throws AuthenticationException {
        if (logger.isTraceEnabled()) {
            logger.tracef("Using DN [%s] for authentication of user", (Object)user.getDn());
        }
        this.operationManager.authenticate(user.getDn().getLdapName(), password);
    }

    @Override
    public void updatePassword(LDAPObject user, String password, LDAPOperationDecorator passwordUpdateDecorator) {
        if (logger.isDebugEnabled()) {
            logger.debugf("Using DN [%s] for updating LDAP password of user", (Object)user.getDn());
        }
        if (this.getConfig().isActiveDirectory()) {
            this.updateADPassword(user.getDn().getLdapName(), password, passwordUpdateDecorator);
            return;
        }
        try {
            if (this.config.useExtendedPasswordModifyOp()) {
                this.operationManager.passwordModifyExtended(user.getDn().getLdapName(), password, passwordUpdateDecorator);
            } else {
                ModificationItem[] mods = new ModificationItem[1];
                BasicAttribute mod0 = new BasicAttribute("userpassword", password);
                mods[0] = new ModificationItem(2, mod0);
                this.operationManager.modifyAttributes(user.getDn().getLdapName(), mods, passwordUpdateDecorator);
            }
        }
        catch (ModelException me) {
            throw me;
        }
        catch (Exception e) {
            throw new ModelException("Error updating password.", (Throwable)e);
        }
    }

    private void updateADPassword(LdapName userDN, String password, LDAPOperationDecorator passwordUpdateDecorator) {
        try {
            String newQuotedPassword = "\"" + password + "\"";
            byte[] newUnicodePassword = newQuotedPassword.getBytes("UTF-16LE");
            BasicAttribute unicodePwd = new BasicAttribute("unicodePwd", newUnicodePassword);
            ArrayList<ModificationItem> modItems = new ArrayList<ModificationItem>();
            modItems.add(new ModificationItem(2, unicodePwd));
            this.operationManager.modifyAttributes(userDN, modItems.toArray(new ModificationItem[0]), passwordUpdateDecorator);
        }
        catch (ModelException me) {
            throw me;
        }
        catch (Exception e) {
            throw new ModelException("Error updating password", (Throwable)e);
        }
    }

    protected Condition createIdentityTypeSearchFilter(LDAPQuery identityQuery) {
        LDAPQueryConditionsBuilder conditionsBuilder = new LDAPQueryConditionsBuilder();
        LinkedHashSet<Condition> conditions = new LinkedHashSet<Condition>(identityQuery.getConditions());
        this.addObjectClassesConditions(conditionsBuilder, identityQuery.getObjectClasses(), conditions);
        return conditionsBuilder.andCondition((Condition[])conditions.toArray(Condition[]::new));
    }

    private Set<Condition> addObjectClassesConditions(LDAPQueryConditionsBuilder conditionsBuilder, Collection<String> objectClasses, Set<Condition> conditions) {
        if (!objectClasses.isEmpty()) {
            for (String objectClass : objectClasses) {
                conditions.add(conditionsBuilder.equal("objectclass", objectClass));
            }
        } else {
            conditions.add(conditionsBuilder.present("objectclass"));
        }
        return conditions;
    }

    private LDAPObject populateAttributedType(SearchResult searchResult, LDAPQuery ldapQuery) {
        Set<String> readOnlyAttrNames = ldapQuery.getReturningReadOnlyLdapAttributes();
        TreeSet<String> lowerCasedAttrNames = new TreeSet<String>();
        for (String attrName : ldapQuery.getReturningLdapAttributes()) {
            lowerCasedAttrNames.add(attrName.toLowerCase());
        }
        try {
            String entryDN = searchResult.getNameInNamespace();
            Attributes attributes = searchResult.getAttributes();
            LDAPObject ldapObject = new LDAPObject();
            LDAPDn dn = LDAPDn.fromString(entryDN);
            ldapObject.setDn(dn);
            ldapObject.setRdnAttributeNames(dn.getFirstRdn().getAllKeys());
            NamingEnumeration<? extends Attribute> ldapAttributes = attributes.getAll();
            while (ldapAttributes.hasMore()) {
                Attribute ldapAttribute = ldapAttributes.next();
                try {
                    ldapAttribute.get();
                }
                catch (NoSuchElementException nsee) {
                    continue;
                }
                String ldapAttributeName = ldapAttribute.getID();
                Matcher m = rangePattern.matcher(ldapAttributeName);
                if (m.matches()) {
                    ldapAttributeName = m.group(1);
                    if (!m.group(3).equals("*")) {
                        try {
                            int max = Integer.parseInt(m.group(3));
                            ldapObject.addRangedAttribute(ldapAttributeName, max);
                        }
                        catch (NumberFormatException e) {
                            logger.warnf("Invalid ranged expresion for attribute: %s", (Object)m.group(0));
                        }
                    }
                }
                if (ldapAttributeName.equalsIgnoreCase(this.getConfig().getUuidLDAPAttributeName())) {
                    Object uuidValue = ldapAttribute.get();
                    ldapObject.setUuid(this.operationManager.decodeEntryUUID(uuidValue));
                }
                if (ldapAttributeName.equalsIgnoreCase(this.getConfig().getUuidLDAPAttributeName()) && !lowerCasedAttrNames.contains(ldapAttributeName.toLowerCase())) continue;
                LinkedHashSet<String> attrValues = new LinkedHashSet<String>();
                NamingEnumeration<?> enumm = ldapAttribute.getAll();
                while (enumm.hasMoreElements()) {
                    String attrVal;
                    Object val = enumm.next();
                    if (val instanceof byte[]) {
                        attrVal = Base64.encodeBytes((byte[])((byte[])val));
                        attrValues.add(attrVal);
                        continue;
                    }
                    attrVal = val.toString().trim();
                    attrValues.add(attrVal);
                }
                if (ldapAttributeName.equalsIgnoreCase("objectclass")) {
                    ldapObject.setObjectClasses(attrValues);
                    continue;
                }
                ldapObject.setAttribute(ldapAttributeName, attrValues);
                if (!readOnlyAttrNames.contains(ldapAttributeName.toLowerCase())) continue;
                ldapObject.addReadOnlyAttributeName(ldapAttributeName);
            }
            if (logger.isTraceEnabled()) {
                logger.tracef("Found ldap object and populated with the attributes. LDAP Object: %s", (Object)ldapObject.toString());
            }
            return ldapObject;
        }
        catch (Exception e) {
            throw new ModelException("Could not populate attribute type " + searchResult.getNameInNamespace() + ".", (Throwable)e);
        }
    }

    protected BasicAttributes extractAttributesForSaving(LDAPObject ldapObject, boolean isCreate) {
        BasicAttributes entryAttributes = new BasicAttributes();
        Set rdnAttrNamesLowerCased = ldapObject.getRdnAttributeNames().stream().map(String::toLowerCase).collect(Collectors.toSet());
        for (Map.Entry<String, Set<String>> attrEntry : ldapObject.getAttributes().entrySet()) {
            String attrName = attrEntry.getKey();
            Set<String> attrValue = attrEntry.getValue();
            if (attrValue == null) {
                logger.warnf("Attribute '%s' is null on LDAP object '%s' . Using empty value to be saved to LDAP", (Object)attrName, (Object)ldapObject.getDn().toString());
                attrValue = Collections.emptySet();
            }
            String attrNameLowercased = attrName.toLowerCase();
            if (isCreate && attrValue.isEmpty() || !isCreate && ldapObject.getReadOnlyAttributeNames().contains(attrNameLowercased) || !isCreate && rdnAttrNamesLowerCased.contains(attrNameLowercased)) continue;
            if (this.getConfig().getBinaryAttributeNames().contains(attrName)) {
                entryAttributes.put(this.createBinaryBasicAttribute(attrName, attrValue));
                continue;
            }
            entryAttributes.put(this.createBasicAttribute(attrName, attrValue));
        }
        if (isCreate) {
            BasicAttribute objectClassAttribute = new BasicAttribute("objectclass");
            for (String objectClassValue : ldapObject.getObjectClasses()) {
                objectClassAttribute.add(objectClassValue);
            }
            entryAttributes.put(objectClassAttribute);
        }
        return entryAttributes;
    }

    private BasicAttribute createBasicAttribute(String attrName, Set<String> attrValue) {
        BasicAttribute attr = new BasicAttribute(attrName);
        for (String value : attrValue) {
            if (value == null || value.trim().length() == 0) {
                value = " ";
            }
            attr.add(value);
        }
        return attr;
    }

    private BasicAttribute createBinaryBasicAttribute(String attrName, Set<String> attrValue) {
        BasicAttribute attr = new BasicAttribute(attrName);
        for (String value : attrValue) {
            if (value == null || value.trim().length() == 0) {
                value = " ";
            }
            try {
                byte[] bytes = Base64.decode((String)value);
                attr.add(bytes);
            }
            catch (IOException ioe) {
                logger.warnf("Wasn't able to Base64 decode the attribute value. Ignoring attribute update. Attribute: %s, Attribute value: %s", (Object)attrName, attrValue);
            }
        }
        return attr;
    }
}

