"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.pplSearchStrategyProvider = void 0;
var _operators = require("rxjs/operators");
var _server = require("../../../data/server");
var _common = require("../../../data/common");
var _utils = require("../../common/utils");
var _utils2 = require("../utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const pplSearchStrategyProvider = (config$, logger, client, usage) => {
  const pplFacet = new _utils2.Facet({
    client,
    logger,
    endpoint: 'enhancements.pplQuery',
    useJobs: false,
    shimResponse: true
  });
  const parseRequest = query => {
    const pipeMap = new Map();
    const pipeArray = query.split('|');
    pipeArray.forEach((pipe, index) => {
      const splitChar = index === 0 ? '=' : ' ';
      const split = pipe.trim().split(splitChar);
      const key = split[0];
      const value = pipe.replace(index === 0 ? `${key}=` : key, '').trim();
      pipeMap.set(key, value);
    });
    const source = pipeMap.get('source');
    const searchQuery = query;
    const filters = pipeMap.get('where');
    const stats = pipeMap.get('stats');
    const aggsQuery = stats ? `source=${source} ${filters ? `| where ${filters}` : ''} | stats ${stats}` : undefined;
    return {
      map: pipeMap,
      search: searchQuery,
      aggs: aggsQuery
    };
  };
  return {
    search: async (context, request, options) => {
      const config = await config$.pipe((0, _operators.first)()).toPromise();
      const uiSettingsClient = await context.core.uiSettings.client;
      const {
        dataFrameHydrationStrategy,
        ...defaultParams
      } = await (0, _server.getDefaultSearchParams)(uiSettingsClient);
      try {
        var _dataFrame$meta;
        const requestParams = parseRequest(request.body.query.qs);
        const source = requestParams === null || requestParams === void 0 ? void 0 : requestParams.map.get('source');
        const {
          schema,
          meta
        } = request.body.df;
        request.body.query = !schema || dataFrameHydrationStrategy === 'perQuery' ? `source=${source} | head` : requestParams.search;
        const rawResponse = await pplFacet.describeQuery(context, request);
        if (!rawResponse.success) {
          return {
            type: _common.DATA_FRAME_TYPES.ERROR,
            body: {
              error: rawResponse.data
            },
            took: rawResponse.took
          };
        }
        const dataFrame = (0, _common.createDataFrame)({
          name: source,
          schema: schema !== null && schema !== void 0 ? schema : rawResponse.data.schema,
          meta,
          fields: (0, _utils.getFields)(rawResponse)
        });
        dataFrame.size = rawResponse.data.datarows.length;
        if (usage) usage.trackSuccess(rawResponse.took);
        if ((_dataFrame$meta = dataFrame.meta) !== null && _dataFrame$meta !== void 0 && _dataFrame$meta.aggsQs) {
          for (const [key, aggQueryString] of Object.entries(dataFrame.meta.aggsQs)) {
            var _rawAggs$data$datarow;
            const aggRequest = parseRequest(aggQueryString);
            const query = aggRequest.aggs;
            request.body.query = query;
            const rawAggs = await pplFacet.describeQuery(context, request);
            dataFrame.aggs = {};
            dataFrame.aggs[key] = (_rawAggs$data$datarow = rawAggs.data.datarows) === null || _rawAggs$data$datarow === void 0 ? void 0 : _rawAggs$data$datarow.map(hit => {
              return {
                key: hit[1],
                value: hit[0]
              };
            });
          }
        }
        return {
          type: _common.DATA_FRAME_TYPES.DEFAULT,
          body: dataFrame,
          took: rawResponse.took
        };
      } catch (e) {
        logger.error(`pplSearchStrategy: ${e.message}`);
        if (usage) usage.trackError();
        throw e;
      }
    }
  };
};
exports.pplSearchStrategyProvider = pplSearchStrategyProvider;