"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _helpers = require("./utils/helpers");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class DestinationsService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "createDestination", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createDestination', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - createDestination:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateDestination", async (context, req, res) => {
      try {
        const {
          destinationId
        } = req.params;
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        const params = {
          body: req.body,
          destinationId,
          ifSeqNo,
          ifPrimaryTerm
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client('alerting.updateDestination', params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - updateDestination:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteDestination", async (context, req, res) => {
      try {
        const {
          destinationId
        } = req.params;
        const params = {
          destinationId
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteDestination', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted'
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - deleteDestination:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getDestination", async (context, req, res) => {
      const {
        destinationId
      } = req.params;
      const client = this.getClientBasedOnDataSource(context, req);
      try {
        const params = {
          destinationId
        };
        const resp = await client('alerting.getDestination', params);
        const destination = resp.destinations[0];
        const version = destination.schema_version;
        const ifSeqNo = destination.seq_no;
        const ifPrimaryTerm = destination.primary_term;
        return res.ok({
          body: {
            ok: true,
            destination,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - getDestination:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getDestinations", async (context, req, res) => {
      const client = this.getClientBasedOnDataSource(context, req);
      const {
        from = 0,
        size = 20,
        search = '',
        sortDirection = 'desc',
        sortField = 'start_time',
        type = 'ALL'
      } = req.query;
      var params;
      switch (sortField) {
        case 'name':
          params = {
            sortString: 'destination.name.keyword',
            sortOrder: sortDirection
          };
          break;
        case 'type':
          params = {
            sortString: 'destination.type',
            sortOrder: sortDirection
          };
          break;
        default:
          params = {};
          break;
      }
      params.startIndex = from;
      params.size = size;
      params.searchString = search;
      if (search.trim()) params.searchString = `*${search.trim().split(' ').join('* *')}*`;
      params.destinationType = type;
      try {
        const resp = await client('alerting.searchDestinations', params);
        const destinations = resp.destinations.map(hit => {
          const destination = hit;
          const id = destination.id;
          const version = destination.schema_version;
          const ifSeqNo = destination.seq_no;
          const ifPrimaryTerm = destination.primary_term;
          return {
            id,
            ...destination,
            version,
            ifSeqNo,
            ifPrimaryTerm
          };
        });
        const totalDestinations = resp.totalDestinations;
        return res.ok({
          body: {
            ok: true,
            destinations,
            totalDestinations
          }
        });
      } catch (err) {
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {}
            }
          });
        }
        return res.ok({
          body: {
            ok: false,
            err: err.message
          }
        });
      }
    });
    /**
     *  -----------------------------------------------------
     *  ----------------- Email Account API -----------------
     *  -----------------------------------------------------
     */
    _defineProperty(this, "createEmailAccount", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createEmailAccount', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - createEmailAccount:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateEmailAccount", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        const params = {
          emailAccountId: id,
          ifSeqNo,
          ifPrimaryTerm,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client('alerting.updateEmailAccount', params);
        const {
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - updateEmailAccount:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteEmailAccount", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          emailAccountId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const deleteResponse = await client('alerting.deleteEmailAccount', params);
        return res.ok({
          body: {
            ok: deleteResponse.result === 'deleted'
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - deleteEmailAccount:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getEmailAccount", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          emailAccountId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getEmailAccount', params);
        const emailAccount = _lodash.default.get(getResponse, 'email_account', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        if (emailAccount) {
          return res.ok({
            body: {
              ok: true,
              resp: emailAccount,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - DestinationService - getEmailAccount:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getEmailAccounts", async (context, req, res) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          sortDirection = 'desc',
          sortField = 'name'
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          must = {
            query_string: {
              default_field: 'email_account.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const sortQueryMap = {
          name: {
            'email_account.name.keyword': sortDirection
          }
        };
        let sort = [];
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) sort = sortQuery;
        const params = {
          body: {
            from,
            size,
            sort,
            query: {
              bool: {
                must
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getEmailAccounts', params);
        const totalEmailAccounts = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const emailAccounts = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source: emailAccount
          } = result;
          return {
            id,
            ...emailAccount,
            ifSeqNo,
            ifPrimaryTerm
          };
        });
        return res.ok({
          body: {
            ok: true,
            emailAccounts,
            totalEmailAccounts
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - getEmailAccounts:', err);
        return res.ok({
          body: {
            ok: false,
            err: err.message
          }
        });
      }
    });
    /**
     *  -----------------------------------------------------
     *  ----------------- Email Group API -------------------
     *  -----------------------------------------------------
     */
    _defineProperty(this, "createEmailGroup", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createEmailGroup', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - createEmailGroup:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateEmailGroup", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        const params = {
          emailGroupId: id,
          ifSeqNo,
          ifPrimaryTerm,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client('alerting.updateEmailGroup', params);
        const {
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - updateEmailGroup:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteEmailGroup", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          emailGroupId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const deleteResponse = await client('alerting.deleteEmailGroup', params);
        return res.ok({
          body: {
            ok: deleteResponse.result === 'deleted'
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - deleteEmailGroup:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getEmailGroup", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          emailGroupId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getEmailGroup', params);
        const emailGroup = _lodash.default.get(getResponse, 'email_group', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        if (emailGroup) {
          return res.ok({
            body: {
              ok: true,
              resp: emailGroup,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - DestinationService - getEmailGroup:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getEmailGroups", async (context, req, res) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          sortDirection = 'desc',
          sortField = 'name'
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          must = {
            query_string: {
              default_field: 'email_group.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const sortQueryMap = {
          name: {
            'email_group.name.keyword': sortDirection
          }
        };
        let sort = [];
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) sort = sortQuery;
        const params = {
          body: {
            from,
            size,
            sort,
            query: {
              bool: {
                must
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getEmailGroups', params);
        const totalEmailGroups = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const emailGroups = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source: emailGroup
          } = result;
          return {
            id,
            ...emailGroup,
            ifSeqNo,
            ifPrimaryTerm
          };
        });
        return res.ok({
          body: {
            ok: true,
            emailGroups,
            totalEmailGroups
          }
        });
      } catch (err) {
        console.error('Alerting - DestinationService - getEmailGroups:', err);
        return res.ok({
          body: {
            ok: false,
            err: err.message
          }
        });
      }
    });
  }
}
exports.default = DestinationsService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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