"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultBackend = void 0;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
var _sample_notebooks = require("../../common/helpers/notebooks/sample_notebooks");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class DefaultBackend {
  constructor() {
    _defineProperty(this, "backend", 'Default Backend');
    // Creates a new notebooks with sample markdown text
    _defineProperty(this, "createNewNotebook", newNotebookName => {
      const noteObject = {
        dateCreated: new Date().toISOString(),
        name: newNotebookName,
        dateModified: new Date().toISOString(),
        backend: 'Default',
        paragraphs: []
      };
      return {
        object: noteObject
      };
    });
    // indexes a notebook with body provided
    _defineProperty(this, "indexNote", async function (client, body) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            notebook: body
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Doc Error:' + error);
      }
    });
    // updates a notebook with updateBody provided as parameter
    _defineProperty(this, "updateNote", async function (client, noteId, updateBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: noteId,
          body: {
            notebook: updateBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Doc Error:' + error);
      }
    });
    // fetched a notebook by Id
    _defineProperty(this, "getNote", async function (client, noteId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: noteId
        });
        if (response.observabilityObjectList.length === 0) {
          throw new Error('notebook id not found');
        }
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Doc Error:' + error);
      }
    });
    // gets first `FETCH_SIZE` notebooks available
    _defineProperty(this, "viewNotes", async function (client, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'notebook'
        });
        return response.observabilityObjectList.map(notebook => ({
          path: notebook.notebook.name,
          id: notebook.objectId,
          dateCreated: notebook.notebook.dateCreated,
          dateModified: notebook.notebook.dateModified
        }));
      } catch (error) {
        if (error.body.error.type === 'index_not_found_exception') {
          return [];
        } else throw new Error('View Notebooks Error:' + error);
      }
    });
    /* Fetches a notebook by id
     * Param: noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "fetchNote", async function (client, noteId, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, noteId);
        return {
          path: noteObject.notebook.name,
          dateCreated: noteObject.notebook.dateCreated,
          dateModified: noteObject.notebook.dateModified,
          paragraphs: noteObject.notebook.paragraphs
        };
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });
    /* Adds a notebook to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addNote", async function (client, params, _wreckOptions) {
      try {
        const newNotebook = this.createNewNotebook(params.name);
        const opensearchClientResponse = await this.indexNote(client, newNotebook.object);
        return {
          status: 'OK',
          message: opensearchClientResponse,
          body: opensearchClientResponse.objectId
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Adds sample notebooks to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addSampleNotes", async function (client, visIds, _wreckOptions) {
      try {
        const notebooks = (0, _sample_notebooks.getSampleNotebooks)(visIds);
        const newNotebooks = [];
        for (let i = 0; i < notebooks.length; i++) {
          const notebook = notebooks[i];
          await this.indexNote(client, notebook.notebook).then(response => {
            newNotebooks.push({
              id: response.objectId,
              name: notebook.notebook.name,
              dateModified: notebook.dateModified,
              dateCreated: notebook.dateCreated
            });
          });
        }
        return {
          status: 'OK',
          message: '',
          body: newNotebooks
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Renames a notebook
     * Params: name -> new name for the notebook to be renamed
     *         noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "renameNote", async function (client, params, _wreckOptions) {
      try {
        const updateNotebook = {
          name: params.name,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          status: 'OK',
          message: opensearchClientResponse
        };
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });
    /* Clone a notebook
     * Params: name -> new name for the cloned notebook
     *         noteId -> Id for the notebook to be cloned
     */
    _defineProperty(this, "cloneNote", async function (client, params, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, params.noteId);
        const newNotebook = this.createNewNotebook(params.name);
        const cloneNotebook = {
          ...newNotebook.object
        };
        cloneNotebook.paragraphs = noteObject.notebook.paragraphs;
        const opensearchClientIndexResponse = await this.indexNote(client, cloneNotebook);
        return {
          status: 'OK',
          body: {
            ...cloneNotebook,
            id: opensearchClientIndexResponse.objectId
          }
        };
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });
    /* Delete a notebook
     * Param: noteId -> Id for the notebook to be deleted
     */
    _defineProperty(this, "deleteNote", async function (client, noteList, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: noteList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });
    /* Export a notebook
     * Param: noteId -> Id for the notebook to be exported
     */
    _defineProperty(this, "exportNote", async function (client, noteId, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, noteId);
        return {
          status: 'OK',
          body: opensearchClientGetResponse
        };
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });
    /* Import a notebook
     * Params: noteObj -> note Object to be imported
     */
    _defineProperty(this, "importNote", async function (client, noteObj, _wreckOptions) {
      try {
        const newNoteObject = {
          ...noteObj
        };
        newNoteObject.id = 'note_' + (0, _uuid.v4)();
        newNoteObject.dateCreated = new Date().toISOString();
        newNoteObject.dateModified = new Date().toISOString();
        const opensearchClientIndexResponse = await this.indexNote(client, newNoteObject);
        return {
          status: 'OK',
          message: opensearchClientIndexResponse,
          body: opensearchClientIndexResponse.objectId
        };
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });
    /* Updates input for required paragraphs
     * Params: paragraphs -> list of paragraphs
     *         paragraphId -> Id of paragraph to be updated
     *         paragraphInput -> Input to be added
     */
    _defineProperty(this, "updateParagraph", function (paragraphs, paragraphId, paragraphInput, paragraphType) {
      try {
        const updatedParagraphs = [];
        paragraphs.map(paragraph => {
          const updatedParagraph = {
            ...paragraph
          };
          if (paragraph.id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            updatedParagraph.input.inputText = paragraphInput;
            if (paragraphType.length > 0) {
              updatedParagraph.input.inputType = paragraphType;
            }
          }
          updatedParagraphs.push(updatedParagraph);
        });
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Update Paragraph Error:' + error);
      }
    });
    // Creates new paragraph with the given input and input type
    _defineProperty(this, "createParagraph", function (paragraphInput, inputType) {
      try {
        let paragraphType = 'MARKDOWN';
        if (inputType === 'VISUALIZATION') {
          paragraphType = 'VISUALIZATION';
        }
        if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          paragraphType = 'OBSERVABILITY_VISUALIZATION';
        }
        if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
          paragraphType = 'QUERY';
        }
        const inputObject = {
          inputType: paragraphType,
          inputText: paragraphInput
        };
        const outputObjects = [{
          outputType: paragraphType,
          result: '',
          execution_time: '0s'
        }];
        const newParagraph = {
          id: 'paragraph_' + (0, _uuid.v4)(),
          dateCreated: new Date().toISOString(),
          dateModified: new Date().toISOString(),
          input: inputObject,
          output: outputObjects
        };
        return newParagraph;
      } catch (error) {
        throw new Error('Create Paragraph Error:' + error);
      }
    });
    /* Runs a paragraph
     * Currently only runs markdown by copying input.inputText to result
     * UI renders Markdown
     */
    _defineProperty(this, "runParagraph", async function (paragraphs, paragraphId, _client) {
      try {
        const updatedParagraphs = [];
        let index = 0;
        for (index = 0; index < paragraphs.length; ++index) {
          const startTime = (0, _performanceNow.default)();
          const updatedParagraph = {
            ...paragraphs[index]
          };
          if (paragraphs[index].id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'QUERY',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'OBSERVABILITY_VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: 'Please select an input type (%sql, %ppl, or %md)',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            }
          }
          updatedParagraphs.push(updatedParagraph);
        }
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Runs it
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content and output result)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, _wreckOptions) {
      try {
        const scopedClient = client.asScoped(request);
        const params = request.body;
        const opensearchClientGetResponse = await this.getNote(scopedClient, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType);
        const updatedOutputParagraphs = await this.runParagraph(updatedInputParagraphs, params.paragraphId, client);
        const updateNotebook = {
          paragraphs: updatedOutputParagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(scopedClient, params.noteId, updateNotebook);
        let resultParagraph = {};
        let index = 0;
        for (index = 0; index < updatedOutputParagraphs.length; ++index) {
          if (params.paragraphId === updatedOutputParagraphs[index].id) {
            resultParagraph = updatedOutputParagraphs[index];
          }
        }
        return resultParagraph;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateFetchParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput);
        const updateNotebook = {
          paragraphs: updatedInputParagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        let resultParagraph = {};
        updatedInputParagraphs.map(paragraph => {
          if (params.paragraphId === paragraph.id) {
            resultParagraph = paragraph;
          }
        });
        return resultParagraph;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });
    /* --> Fetches the Paragraph
     * --> Adds a Paragraph with input content
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be fetched
     */
    _defineProperty(this, "addFetchNewParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const paragraphs = opensearchClientGetResponse.notebook.paragraphs;
        const newParagraph = this.createParagraph(params.paragraphInput, params.inputType);
        paragraphs.splice(params.paragraphIndex, 0, newParagraph);
        const updateNotebook = {
          paragraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return newParagraph;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });
    /* --> Deletes a Paragraph with id
     * --> Fetches the all other Paragraphs as a list
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be deleted
     */
    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        if (params.paragraphId !== undefined) {
          opensearchClientGetResponse.notebook.paragraphs.map(paragraph => {
            if (paragraph.id !== params.paragraphId) {
              updatedparagraphs.push(paragraph);
            }
          });
        }
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        console.log('error', error);
        throw new Error('Delete Paragraph Error:' + error);
      }
    });
    /* --> Clears output for all the paragraphs
     * --> Fetches the all Paragraphs as a list (with cleared outputs)
     * --> Updates the notebook in index
     * Param: noteId -> Id of notebook to be cleared
     */
    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        opensearchClientGetResponse.notebook.paragraphs.map(paragraph => {
          const updatedParagraph = {
            ...paragraph
          };
          updatedParagraph.output = [];
          updatedparagraphs.push(updatedParagraph);
        });
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  }
}
exports.DefaultBackend = DefaultBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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