/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cache.common.tier;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.ToLongBiFunction;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.cache.common.policy.TookTimePolicy;
import org.opensearch.cache.common.tier.TieredSpilloverCacheSettings;
import org.opensearch.cache.common.tier.TieredSpilloverCacheStatsHolder;
import org.opensearch.common.annotation.ExperimentalApi;
import org.opensearch.common.cache.CacheType;
import org.opensearch.common.cache.ICache;
import org.opensearch.common.cache.ICacheKey;
import org.opensearch.common.cache.LoadAwareCacheLoader;
import org.opensearch.common.cache.RemovalListener;
import org.opensearch.common.cache.RemovalNotification;
import org.opensearch.common.cache.RemovalReason;
import org.opensearch.common.cache.stats.ImmutableCacheStatsHolder;
import org.opensearch.common.cache.store.config.CacheConfig;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.settings.Setting;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.util.concurrent.ReleasableLock;

@ExperimentalApi
public class TieredSpilloverCache<K, V>
implements ICache<K, V> {
    private static final List<RemovalReason> SPILLOVER_REMOVAL_REASONS = List.of(RemovalReason.EVICTED, RemovalReason.CAPACITY);
    private static final Logger logger = LogManager.getLogger(TieredSpilloverCache.class);
    private final ICache<K, V> diskCache;
    private final ICache<K, V> onHeapCache;
    private final RemovalListener<ICacheKey<K>, V> onDiskRemovalListener;
    private final RemovalListener<ICacheKey<K>, V> onHeapRemovalListener;
    private final RemovalListener<ICacheKey<K>, V> removalListener;
    private final TieredSpilloverCacheStatsHolder statsHolder;
    private ToLongBiFunction<ICacheKey<K>, V> weigher;
    private final List<String> dimensionNames;
    ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    ReleasableLock readLock = new ReleasableLock(this.readWriteLock.readLock());
    ReleasableLock writeLock = new ReleasableLock(this.readWriteLock.writeLock());
    private final Map<ICache<K, V>, TierInfo> caches;
    private final List<Predicate<V>> policies;
    Map<ICacheKey<K>, CompletableFuture<Tuple<ICacheKey<K>, V>>> completableFutureMap = new ConcurrentHashMap<ICacheKey<K>, CompletableFuture<Tuple<ICacheKey<K>, V>>>();

    TieredSpilloverCache(Builder<K, V> builder) {
        Objects.requireNonNull(builder.onHeapCacheFactory, "onHeap cache builder can't be null");
        Objects.requireNonNull(builder.diskCacheFactory, "disk cache builder can't be null");
        Objects.requireNonNull(builder.cacheConfig, "cache config can't be null");
        Objects.requireNonNull(builder.cacheConfig.getClusterSettings(), "cluster settings can't be null");
        this.removalListener = Objects.requireNonNull(builder.removalListener, "Removal listener can't be null");
        this.onHeapRemovalListener = new HeapTierRemovalListener(this);
        this.onDiskRemovalListener = new DiskTierRemovalListener(this);
        this.weigher = Objects.requireNonNull(builder.cacheConfig.getWeigher(), "Weigher can't be null");
        this.onHeapCache = builder.onHeapCacheFactory.create(new CacheConfig.Builder().setRemovalListener(this.onHeapRemovalListener).setKeyType(builder.cacheConfig.getKeyType()).setValueType(builder.cacheConfig.getValueType()).setSettings(builder.cacheConfig.getSettings()).setWeigher(builder.cacheConfig.getWeigher()).setDimensionNames(builder.cacheConfig.getDimensionNames()).setMaxSizeInBytes(builder.cacheConfig.getMaxSizeInBytes().longValue()).setExpireAfterAccess(builder.cacheConfig.getExpireAfterAccess()).setClusterSettings(builder.cacheConfig.getClusterSettings()).setStatsTrackingEnabled(false).build(), builder.cacheType, builder.cacheFactories);
        this.diskCache = builder.diskCacheFactory.create(new CacheConfig.Builder().setRemovalListener(this.onDiskRemovalListener).setKeyType(builder.cacheConfig.getKeyType()).setValueType(builder.cacheConfig.getValueType()).setSettings(builder.cacheConfig.getSettings()).setWeigher(builder.cacheConfig.getWeigher()).setKeySerializer(builder.cacheConfig.getKeySerializer()).setValueSerializer(builder.cacheConfig.getValueSerializer()).setDimensionNames(builder.cacheConfig.getDimensionNames()).setStatsTrackingEnabled(false).build(), builder.cacheType, builder.cacheFactories);
        Boolean isDiskCacheEnabled = (Boolean)TieredSpilloverCacheSettings.DISK_CACHE_ENABLED_SETTING_MAP.get(builder.cacheType).get(builder.cacheConfig.getSettings());
        LinkedHashMap<ICache<K, V>, TierInfo> cacheListMap = new LinkedHashMap<ICache<K, V>, TierInfo>();
        cacheListMap.put(this.onHeapCache, new TierInfo(true, "on_heap"));
        cacheListMap.put(this.diskCache, new TierInfo(isDiskCacheEnabled, "disk"));
        this.caches = Collections.synchronizedMap(cacheListMap);
        this.dimensionNames = builder.cacheConfig.getDimensionNames();
        this.statsHolder = new TieredSpilloverCacheStatsHolder(this.dimensionNames, isDiskCacheEnabled);
        this.policies = builder.policies;
        builder.cacheConfig.getClusterSettings().addSettingsUpdateConsumer(TieredSpilloverCacheSettings.DISK_CACHE_ENABLED_SETTING_MAP.get(builder.cacheType), this::enableDisableDiskCache);
    }

    ICache<K, V> getOnHeapCache() {
        return this.onHeapCache;
    }

    ICache<K, V> getDiskCache() {
        return this.diskCache;
    }

    void enableDisableDiskCache(Boolean isDiskCacheEnabled) {
        this.caches.put(this.diskCache, new TierInfo(isDiskCacheEnabled, "disk"));
        this.statsHolder.setDiskCacheEnabled(isDiskCacheEnabled);
    }

    public V get(ICacheKey<K> key) {
        Tuple<V, String> cacheValueTuple = this.getValueFromTieredCache(true).apply(key);
        if (cacheValueTuple == null) {
            return null;
        }
        return (V)cacheValueTuple.v1();
    }

    public void put(ICacheKey<K> key, V value) {
        try (ReleasableLock ignore = this.writeLock.acquire();){
            this.onHeapCache.put(key, value);
            this.updateStatsOnPut("on_heap", key, value);
        }
    }

    public V computeIfAbsent(ICacheKey<K> key, LoadAwareCacheLoader<ICacheKey<K>, V> loader) throws Exception {
        Tuple<V, String> cacheValueTuple;
        CompletableFuture<Tuple<ICacheKey<K>, V>> future = null;
        try (ReleasableLock ignore = this.readLock.acquire();){
            cacheValueTuple = this.getValueFromTieredCache(false).apply(key);
            if (cacheValueTuple == null) {
                future = this.completableFutureMap.putIfAbsent(key, new CompletableFuture());
            }
        }
        List<String> heapDimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, "on_heap");
        List<String> diskDimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, "disk");
        if (cacheValueTuple == null) {
            V value = this.compute(key, loader, future);
            if (loader.isLoaded()) {
                this.updateStatsOnPut("on_heap", key, value);
                this.statsHolder.incrementMisses(heapDimensionValues);
                if (this.caches.get(this.diskCache).isEnabled()) {
                    this.statsHolder.incrementMisses(diskDimensionValues);
                }
            } else {
                this.statsHolder.incrementHits(heapDimensionValues);
            }
            return value;
        }
        if (((String)cacheValueTuple.v2()).equals("on_heap")) {
            this.statsHolder.incrementHits(heapDimensionValues);
        } else if (((String)cacheValueTuple.v2()).equals("disk")) {
            this.statsHolder.incrementMisses(heapDimensionValues);
            this.statsHolder.incrementHits(diskDimensionValues);
        }
        return (V)cacheValueTuple.v1();
    }

    private V compute(ICacheKey<K> key, LoadAwareCacheLoader<ICacheKey<K>, V> loader, CompletableFuture<Tuple<ICacheKey<K>, V>> future) throws Exception {
        BiFunction<Tuple, Throwable, Void> handler = (pair, ex) -> {
            if (pair != null) {
                try (ReleasableLock ignore = this.writeLock.acquire();){
                    this.onHeapCache.put((ICacheKey)pair.v1(), pair.v2());
                }
                catch (Exception e) {
                    logger.warn("Exception occurred while putting item onto heap cache", (Throwable)e);
                }
            } else if (ex != null) {
                logger.warn("Exception occurred while trying to compute the value", ex);
            }
            this.completableFutureMap.remove(key);
            return null;
        };
        Object value = null;
        if (future == null) {
            future = this.completableFutureMap.get(key);
            future.handle(handler);
            try {
                value = loader.load(key);
            }
            catch (Exception ex2) {
                future.completeExceptionally(ex2);
                throw new ExecutionException(ex2);
            }
            if (value == null) {
                NullPointerException npe = new NullPointerException("Loader returned a null value");
                future.completeExceptionally(npe);
                throw new ExecutionException(npe);
            }
            future.complete(new Tuple(key, value));
        } else {
            try {
                value = future.get().v2();
            }
            catch (InterruptedException ex3) {
                throw new IllegalStateException(ex3);
            }
        }
        return (V)value;
    }

    public void invalidate(ICacheKey<K> key) {
        for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
            if (key.getDropStatsForDimensions()) {
                List<String> dimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, cacheEntry.getValue().tierName);
                this.statsHolder.removeDimensions(dimensionValues);
            }
            if (key.key == null) continue;
            ReleasableLock ignore = this.writeLock.acquire();
            try {
                cacheEntry.getKey().invalidate(key);
            }
            finally {
                if (ignore == null) continue;
                ignore.close();
            }
        }
    }

    public void invalidateAll() {
        try (ReleasableLock ignore = this.writeLock.acquire();){
            for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
                cacheEntry.getKey().invalidateAll();
            }
        }
        this.statsHolder.reset();
    }

    public Iterable<ICacheKey<K>> keys() {
        ArrayList<Iterable> iterableList = new ArrayList<Iterable>();
        for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
            iterableList.add(cacheEntry.getKey().keys());
        }
        Iterable[] iterables = iterableList.toArray(new Iterable[0]);
        return new ConcatenatedIterables<ICacheKey<K>>(iterables);
    }

    public long count() {
        return this.statsHolder.count();
    }

    public void refresh() {
        try (ReleasableLock ignore = this.writeLock.acquire();){
            for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
                cacheEntry.getKey().refresh();
            }
        }
    }

    public void close() throws IOException {
        for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
            cacheEntry.getKey().close();
        }
    }

    public ImmutableCacheStatsHolder stats(String[] levels) {
        return this.statsHolder.getImmutableCacheStatsHolder(levels);
    }

    private Function<ICacheKey<K>, Tuple<V, String>> getValueFromTieredCache(boolean captureStats) {
        return key -> {
            try (ReleasableLock ignore = this.readLock.acquire();){
                for (Map.Entry<ICache<K, V>, TierInfo> cacheEntry : this.caches.entrySet()) {
                    if (!cacheEntry.getValue().isEnabled()) continue;
                    Object value = cacheEntry.getKey().get(key);
                    String tierValue = cacheEntry.getValue().tierName;
                    List<String> dimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, tierValue);
                    if (value != null) {
                        if (captureStats) {
                            this.statsHolder.incrementHits(dimensionValues);
                        }
                        Tuple tuple = new Tuple(value, (Object)tierValue);
                        return tuple;
                    }
                    if (!captureStats) continue;
                    this.statsHolder.incrementMisses(dimensionValues);
                }
                Iterator<Map.Entry<ICache<K, V>, TierInfo>> iterator = null;
                return iterator;
            }
        };
    }

    void handleRemovalFromHeapTier(RemovalNotification<ICacheKey<K>, V> notification) {
        boolean canCacheOnDisk;
        ICacheKey key = (ICacheKey)notification.getKey();
        boolean wasEvicted = SPILLOVER_REMOVAL_REASONS.contains(notification.getRemovalReason());
        boolean countEvictionTowardsTotal = false;
        boolean exceptionOccurredOnDiskCachePut = false;
        boolean bl = canCacheOnDisk = this.caches.get(this.diskCache).isEnabled() && wasEvicted && this.evaluatePolicies(notification.getValue());
        if (canCacheOnDisk) {
            try (ReleasableLock ignore = this.writeLock.acquire();){
                this.diskCache.put(key, notification.getValue());
            }
            catch (Exception ex) {
                logger.warn("Exception occurred while putting item to disk cache", (Throwable)ex);
                exceptionOccurredOnDiskCachePut = true;
            }
            if (!exceptionOccurredOnDiskCachePut) {
                this.updateStatsOnPut("disk", key, notification.getValue());
            }
        }
        if (!canCacheOnDisk || exceptionOccurredOnDiskCachePut) {
            this.removalListener.onRemoval(notification);
            countEvictionTowardsTotal = true;
        }
        this.updateStatsOnRemoval("on_heap", wasEvicted, key, notification.getValue(), countEvictionTowardsTotal);
    }

    void handleRemovalFromDiskTier(RemovalNotification<ICacheKey<K>, V> notification) {
        this.removalListener.onRemoval(notification);
        boolean wasEvicted = SPILLOVER_REMOVAL_REASONS.contains(notification.getRemovalReason());
        this.updateStatsOnRemoval("disk", wasEvicted, (ICacheKey)notification.getKey(), notification.getValue(), true);
    }

    void updateStatsOnRemoval(String removedFromTierValue, boolean wasEvicted, ICacheKey<K> key, V value, boolean countEvictionTowardsTotal) {
        List<String> dimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, removedFromTierValue);
        if (wasEvicted) {
            this.statsHolder.incrementEvictions(dimensionValues, countEvictionTowardsTotal);
        }
        this.statsHolder.decrementItems(dimensionValues);
        this.statsHolder.decrementSizeInBytes(dimensionValues, this.weigher.applyAsLong(key, value));
    }

    void updateStatsOnPut(String destinationTierValue, ICacheKey<K> key, V value) {
        List<String> dimensionValues = this.statsHolder.getDimensionsWithTierValue(key.dimensions, destinationTierValue);
        this.statsHolder.incrementItems(dimensionValues);
        this.statsHolder.incrementSizeInBytes(dimensionValues, this.weigher.applyAsLong(key, value));
    }

    boolean evaluatePolicies(V value) {
        for (Predicate<V> policy : this.policies) {
            if (policy.test(value)) continue;
            return false;
        }
        return true;
    }

    public static class Builder<K, V> {
        private ICache.Factory onHeapCacheFactory;
        private ICache.Factory diskCacheFactory;
        private RemovalListener<ICacheKey<K>, V> removalListener;
        private CacheConfig<K, V> cacheConfig;
        private CacheType cacheType;
        private Map<String, ICache.Factory> cacheFactories;
        private final ArrayList<Predicate<V>> policies = new ArrayList();

        public Builder<K, V> setOnHeapCacheFactory(ICache.Factory onHeapCacheFactory) {
            this.onHeapCacheFactory = onHeapCacheFactory;
            return this;
        }

        public Builder<K, V> setDiskCacheFactory(ICache.Factory diskCacheFactory) {
            this.diskCacheFactory = diskCacheFactory;
            return this;
        }

        public Builder<K, V> setRemovalListener(RemovalListener<ICacheKey<K>, V> removalListener) {
            this.removalListener = removalListener;
            return this;
        }

        public Builder<K, V> setCacheConfig(CacheConfig<K, V> cacheConfig) {
            this.cacheConfig = cacheConfig;
            return this;
        }

        public Builder<K, V> setCacheType(CacheType cacheType) {
            this.cacheType = cacheType;
            return this;
        }

        public Builder<K, V> setCacheFactories(Map<String, ICache.Factory> cacheFactories) {
            this.cacheFactories = cacheFactories;
            return this;
        }

        public Builder<K, V> addPolicy(Predicate<V> policy) {
            this.policies.add(policy);
            return this;
        }

        public Builder<K, V> addPolicies(List<Predicate<V>> policies) {
            this.policies.addAll(policies);
            return this;
        }

        public TieredSpilloverCache<K, V> build() {
            return new TieredSpilloverCache(this);
        }
    }

    private class HeapTierRemovalListener
    implements RemovalListener<ICacheKey<K>, V> {
        private final TieredSpilloverCache<K, V> tsc;

        HeapTierRemovalListener(TieredSpilloverCache<K, V> tsc) {
            this.tsc = tsc;
        }

        public void onRemoval(RemovalNotification<ICacheKey<K>, V> notification) {
            this.tsc.handleRemovalFromHeapTier(notification);
        }
    }

    private class DiskTierRemovalListener
    implements RemovalListener<ICacheKey<K>, V> {
        private final TieredSpilloverCache<K, V> tsc;

        DiskTierRemovalListener(TieredSpilloverCache<K, V> tsc) {
            this.tsc = tsc;
        }

        public void onRemoval(RemovalNotification<ICacheKey<K>, V> notification) {
            this.tsc.handleRemovalFromDiskTier(notification);
        }
    }

    private class TierInfo {
        AtomicBoolean isEnabled;
        final String tierName;

        TierInfo(boolean isEnabled, String tierName) {
            this.isEnabled = new AtomicBoolean(isEnabled);
            this.tierName = tierName;
        }

        boolean isEnabled() {
            return this.isEnabled.get();
        }
    }

    static class ConcatenatedIterables<K>
    implements Iterable<K> {
        final Iterable<K>[] iterables;

        ConcatenatedIterables(Iterable<K>[] iterables) {
            this.iterables = iterables;
        }

        @Override
        public Iterator<K> iterator() {
            Iterator[] iterators = new Iterator[this.iterables.length];
            for (int i = 0; i < this.iterables.length; ++i) {
                iterators[i] = this.iterables[i].iterator();
            }
            return new ConcatenatedIterator(iterators);
        }

        static class ConcatenatedIterator<T>
        implements Iterator<T> {
            private final Iterator<T>[] iterators;
            private int currentIteratorIndex;
            private Iterator<T> currentIterator;

            public ConcatenatedIterator(Iterator<T>[] iterators) {
                this.iterators = iterators;
                this.currentIteratorIndex = 0;
                this.currentIterator = iterators[this.currentIteratorIndex];
            }

            @Override
            public boolean hasNext() {
                while (!this.currentIterator.hasNext()) {
                    ++this.currentIteratorIndex;
                    if (this.currentIteratorIndex == this.iterators.length) {
                        return false;
                    }
                    this.currentIterator = this.iterators[this.currentIteratorIndex];
                }
                return true;
            }

            @Override
            public T next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                return this.currentIterator.next();
            }

            @Override
            public void remove() {
                this.currentIterator.remove();
            }
        }
    }

    public static class TieredSpilloverCacheFactory
    implements ICache.Factory {
        public static final String TIERED_SPILLOVER_CACHE_NAME = "tiered_spillover";

        public <K, V> ICache<K, V> create(CacheConfig<K, V> config, CacheType cacheType, Map<String, ICache.Factory> cacheFactories) {
            Settings settings = config.getSettings();
            Setting onHeapSetting = TieredSpilloverCacheSettings.TIERED_SPILLOVER_ONHEAP_STORE_NAME.getConcreteSettingForNamespace(cacheType.getSettingPrefix());
            String onHeapCacheStoreName = (String)onHeapSetting.get(settings);
            if (!cacheFactories.containsKey(onHeapCacheStoreName)) {
                throw new IllegalArgumentException("No associated onHeapCache found for tieredSpilloverCache for cacheType:" + String.valueOf(cacheType));
            }
            ICache.Factory onHeapCacheFactory = cacheFactories.get(onHeapCacheStoreName);
            Setting onDiskSetting = TieredSpilloverCacheSettings.TIERED_SPILLOVER_DISK_STORE_NAME.getConcreteSettingForNamespace(cacheType.getSettingPrefix());
            String diskCacheStoreName = (String)onDiskSetting.get(settings);
            if (!cacheFactories.containsKey(diskCacheStoreName)) {
                throw new IllegalArgumentException("No associated diskCache found for tieredSpilloverCache for cacheType:" + String.valueOf(cacheType));
            }
            ICache.Factory diskCacheFactory = cacheFactories.get(diskCacheStoreName);
            TimeValue diskPolicyThreshold = (TimeValue)TieredSpilloverCacheSettings.TOOK_TIME_POLICY_CONCRETE_SETTINGS_MAP.get(cacheType).get(settings);
            Function cachedResultParser = Objects.requireNonNull(config.getCachedResultParser(), "Cached result parser fn can't be null");
            return new Builder().setDiskCacheFactory(diskCacheFactory).setOnHeapCacheFactory(onHeapCacheFactory).setRemovalListener(config.getRemovalListener()).setCacheConfig(config).setCacheType(cacheType).addPolicy(new TookTimePolicy(diskPolicyThreshold, cachedResultParser, config.getClusterSettings(), cacheType)).build();
        }

        public String getCacheName() {
            return TIERED_SPILLOVER_CACHE_NAME;
        }
    }
}

