import encoding.leb128
import encoding.hex

struct PairU {
	value   u64
	encoded string
}

struct PairI {
	value   i64
	encoded string
}

fn test_basic() {
	assert leb128.encode_u64(624485) == [u8(0xe5), 0x8e, 0x26]
	uval, ulen := leb128.decode_u64([u8(0xe5), 0x8e, 0x26])
	assert uval == 624485, 'val, _ := leb128.decode_u64([u8(0xe5), 0x8e, 0x26]) == 624486'
	assert ulen == 3, '_, len := leb128.decode_u64([u8(0xe5), 0x8e, 0x26]) == 3'
	assert leb128.encode_i64(-123456) == [u8(0xc0), 0xbb, 0x78]
	sval, slen := leb128.decode_i64([u8(0xc0), 0xbb, 0x78])
	assert sval == -123456, 'val, _ := leb128.decode_i64([u8(0xc0), 0xbb, 0x78]) == -123456'
	assert slen == 3, '_, len := leb128.decode_i64([u8(0xc0), 0xbb, 0x78]) == 3'
}

fn test_unsigned_data() {
	for x in unsigned_data {
		assert leb128.encode_u64(x.value).hex() == x.encoded, 'leb128.encode_u64( ${x.value} )  == ${x.encoded}'
		bytes := hex.decode(x.encoded)!
		// eprintln('>> bytes: ${bytes} | pair: ${x}')
		val, len := leb128.decode_u64(bytes)
		assert val == x.value, 'val, _ := leb128.decode_u64( ${x.encoded} ) == ${x.value}'
		assert len == bytes.len, '_, len := leb128.decode_u64( ${x.encoded} ) = ${bytes.len}'
	}
}

fn test_signed_data() {
	for x in signed_data {
		assert leb128.encode_i64(x.value).hex() == x.encoded, 'k: ${x.value} | v: ${x.encoded}'
		bytes := hex.decode(x.encoded)!
		// eprintln('>> bytes: ${bytes} | pair: ${x}')
		val, len := leb128.decode_i64(bytes)
		assert val == x.value, 'val, _ := leb128.decode_i64( ${x.encoded} ) == ${x.value}'
		assert len == bytes.len, '_, len := leb128.decode_i64( ${x.encoded} ) == ${x.value}'
	}
}

const unsigned_data = [
	PairU{217, 'd901'},
	PairU{34, '22'},
	PairU{233, 'e901'},
	PairU{51, '33'},
	PairU{69, '45'},
	PairU{233, 'e901'},
	PairU{150, '9601'},
	PairU{149, '9501'},
	PairU{59, '3b'},
	PairU{207, 'cf01'},
	PairU{189, 'bd01'},
	PairU{124, '7c'},
	PairU{13, '0d'},
	PairU{150, '9601'},
	PairU{14, '0e'},
	PairU{19, '13'},
	PairU{80, '50'},
	PairU{172, 'ac01'},
	PairU{252, 'fc01'},
	PairU{89, '59'},
	PairU{28, '1c'},
	PairU{55, '37'},
	PairU{59, '3b'},
	PairU{208, 'd001'},
	PairU{114, '72'},
	PairU{230, 'e601'},
	PairU{225, 'e101'},
	PairU{228, 'e401'},
	PairU{158, '9e01'},
	PairU{29, '1d'},
	PairU{232, 'e801'},
	PairU{169, 'a901'},
	PairU{195, 'c301'},
	PairU{13, '0d'},
	PairU{176, 'b001'},
	PairU{208, 'd001'},
	PairU{47, '2f'},
	PairU{105, '69'},
	PairU{170, 'aa01'},
	PairU{14, '0e'},
	PairU{180, 'b401'},
	PairU{66, '42'},
	PairU{112, '70'},
	PairU{172, 'ac01'},
	PairU{145, '9101'},
	PairU{114, '72'},
	PairU{141, '8d01'},
	PairU{254, 'fe01'},
	PairU{184, 'b801'},
	PairU{28, '1c'},
	PairU{217, 'd901'},
	PairU{74, '4a'},
	PairU{109, '6d'},
	PairU{123, '7b'},
	PairU{11, '0b'},
	PairU{34, '22'},
	PairU{85, '55'},
	PairU{134, '8601'},
	PairU{200, 'c801'},
	PairU{156, '9c01'},
	PairU{4, '04'},
	PairU{91, '5b'},
	PairU{114, '72'},
	PairU{47, '2f'},
	PairU{200, 'c801'},
	PairU{77, '4d'},
	PairU{98, '62'},
	PairU{7, '07'},
	PairU{26, '1a'},
	PairU{100, '64'},
	PairU{100, '64'},
	PairU{132, '8401'},
	PairU{118, '76'},
	PairU{205, 'cd01'},
	PairU{251, 'fb01'},
	PairU{26, '1a'},
	PairU{203, 'cb01'},
	PairU{252, 'fc01'},
	PairU{29, '1d'},
	PairU{182, 'b601'},
	PairU{221, 'dd01'},
	PairU{181, 'b501'},
	PairU{227, 'e301'},
	PairU{43, '2b'},
	PairU{68, '44'},
	PairU{68, '44'},
	PairU{147, '9301'},
	PairU{209, 'd101'},
	PairU{37, '25'},
	PairU{125, '7d'},
	PairU{183, 'b701'},
	PairU{215, 'd701'},
	PairU{82, '52'},
	PairU{197, 'c501'},
	PairU{238, 'ee01'},
	PairU{83, '53'},
	PairU{27, '1b'},
	PairU{191, 'bf01'},
	PairU{125, '7d'},
	PairU{26, '1a'},
	PairU{132, '8401'},
	PairU{138, '8a01'},
	PairU{9, '09'},
	PairU{32, '20'},
	PairU{116, '74'},
	PairU{0, '00'},
	PairU{19, '13'},
	PairU{179, 'b301'},
	PairU{120, '78'},
	PairU{9, '09'},
	PairU{21, '15'},
	PairU{223, 'df01'},
	PairU{83, '53'},
	PairU{218, 'da01'},
	PairU{175, 'af01'},
	PairU{131, '8301'},
	PairU{162, 'a201'},
	PairU{138, '8a01'},
	PairU{196, 'c401'},
	PairU{108, '6c'},
	PairU{41, '29'},
	PairU{6, '06'},
	PairU{97, '61'},
	PairU{171, 'ab01'},
	PairU{42, '2a'},
	PairU{99, '63'},
	PairU{141, '8d01'},
	PairU{253, 'fd01'},
	PairU{122, '7a'},
	PairU{139, '8b01'},
	PairU{143, '8f01'},
	PairU{12, '0c'},
	PairU{150, '9601'},
	PairU{163, 'a301'},
	PairU{195, 'c301'},
	PairU{132, '8401'},
	PairU{25, '19'},
	PairU{4, '04'},
	PairU{119, '77'},
	PairU{102, '66'},
	PairU{67, '43'},
	PairU{91, '5b'},
	PairU{40, '28'},
	PairU{32, '20'},
	PairU{77, '4d'},
	PairU{90, '5a'},
	PairU{45, '2d'},
	PairU{51, '33'},
	PairU{46, '2e'},
	PairU{10, '0a'},
	PairU{153, '9901'},
	PairU{234, 'ea01'},
	PairU{71, '47'},
	PairU{170, 'aa01'},
	PairU{113, '71'},
	PairU{152, '9801'},
	PairU{209, 'd101'},
	PairU{131, '8301'},
	PairU{27, '1b'},
	PairU{117, '75'},
	PairU{151, '9701'},
	PairU{240, 'f001'},
	PairU{142, '8e01'},
	PairU{248, 'f801'},
	PairU{214, 'd601'},
	PairU{177, 'b101'},
	PairU{187, 'bb01'},
	PairU{59, '3b'},
	PairU{249, 'f901'},
	PairU{243, 'f301'},
	PairU{250, 'fa01'},
	PairU{0, '00'},
	PairU{178, 'b201'},
	PairU{189, 'bd01'},
	PairU{175, 'af01'},
	PairU{161, 'a101'},
	PairU{255, 'ff01'},
	PairU{109, '6d'},
	PairU{10, '0a'},
	PairU{8, '08'},
	PairU{74, '4a'},
	PairU{183, 'b701'},
	PairU{232, 'e801'},
	PairU{189, 'bd01'},
	PairU{200, 'c801'},
	PairU{54, '36'},
	PairU{51, '33'},
	PairU{136, '8801'},
	PairU{152, '9801'},
	PairU{243, 'f301'},
	PairU{38, '26'},
	PairU{222, 'de01'},
	PairU{195, 'c301'},
	PairU{11, '0b'},
	PairU{199, 'c701'},
	PairU{220, 'dc01'},
	PairU{91, '5b'},
	PairU{96, '60'},
	PairU{158, '9e01'},
	PairU{246, 'f601'},
	PairU{14, '0e'},
	PairU{178, 'b201'},
	PairU{120, '78'},
	PairU{242, 'f201'},
	PairU{48, '30'},
	PairU{189, 'bd01'},
	PairU{121, '79'},
	PairU{215, 'd701'},
	PairU{124, '7c'},
	PairU{197, 'c501'},
	PairU{193, 'c101'},
	PairU{8, '08'},
	PairU{231, 'e701'},
	PairU{135, '8701'},
	PairU{32, '20'},
	PairU{102, '66'},
	PairU{8, '08'},
	PairU{16, '10'},
	PairU{221, 'dd01'},
	PairU{100, '64'},
	PairU{140, '8c01'},
	PairU{98, '62'},
	PairU{66, '42'},
	PairU{159, '9f01'},
	PairU{6, '06'},
	PairU{191, 'bf01'},
	PairU{255, 'ff01'},
	PairU{208, 'd001'},
	PairU{132, '8401'},
	PairU{215, 'd701'},
	PairU{218, 'da01'},
	PairU{72, '48'},
	PairU{23, '17'},
	PairU{18, '12'},
	PairU{146, '9201'},
	PairU{118, '76'},
	PairU{113, '71'},
	PairU{76, '4c'},
	PairU{207, 'cf01'},
	PairU{145, '9101'},
	PairU{47, '2f'},
	PairU{35, '23'},
	PairU{164, 'a401'},
	PairU{114, '72'},
	PairU{243, 'f301'},
	PairU{63, '3f'},
	PairU{241, 'f101'},
	PairU{221, 'dd01'},
	PairU{14, '0e'},
	PairU{81, '51'},
	PairU{155, '9b01'},
	PairU{121, '79'},
	PairU{157, '9d01'},
	PairU{119, '77'},
	PairU{218, 'da01'},
	PairU{63, '3f'},
	PairU{18538, 'ea9001'},
	PairU{12035, '835e'},
	PairU{23481, 'b9b701'},
	PairU{55732, 'b4b303'},
	PairU{37710, 'cea602'},
	PairU{36807, 'c79f02'},
	PairU{43930, '9ad702'},
	PairU{19926, 'd69b01'},
	PairU{17469, 'bd8801'},
	PairU{14134, 'b66e'},
	PairU{9471, 'ff49'},
	PairU{29119, 'bfe301'},
	PairU{1869, 'cd0e'},
	PairU{12136, 'e85e'},
	PairU{38328, 'b8ab02'},
	PairU{8640, 'c043'},
	PairU{46918, 'c6ee02'},
	PairU{55875, 'c3b403'},
	PairU{670, '9e05'},
	PairU{60244, 'd4d603'},
	PairU{27684, 'a4d801'},
	PairU{52310, 'd69803'},
	PairU{50940, 'fc8d03'},
	PairU{10515, '9352'},
	PairU{17383, 'e78701'},
	PairU{43084, 'ccd002'},
	PairU{48432, 'b0fa02'},
	PairU{60020, 'f4d403'},
	PairU{1860, 'c40e'},
	PairU{49084, 'bcff02'},
	PairU{15937, 'c17c'},
	PairU{21131, '8ba501'},
	PairU{31846, 'e6f801'},
	PairU{33438, '9e8502'},
	PairU{32143, '8ffb01'},
	PairU{58562, 'c2c903'},
	PairU{22492, 'dcaf01'},
	PairU{40033, 'e1b802'},
	PairU{48388, '84fa02'},
	PairU{29290, 'eae401'},
	PairU{53800, 'a8a403'},
	PairU{57378, 'a2c003'},
	PairU{47673, 'b9f402'},
	PairU{59138, '82ce03'},
	PairU{37202, 'd2a202'},
	PairU{49063, 'a7ff02'},
	PairU{47819, 'cbf502'},
	PairU{52832, 'e09c03'},
	PairU{60236, 'ccd603'},
	PairU{55375, 'cfb003'},
	PairU{7663, 'ef3b'},
	PairU{7074, 'a237'},
	PairU{57990, '86c503'},
	PairU{18953, '899401'},
	PairU{5181, 'bd28'},
	PairU{9014, 'b646'},
	PairU{10697, 'c953'},
	PairU{63591, 'e7f003'},
	PairU{46769, 'b1ed02'},
	PairU{56941, 'edbc03'},
	PairU{42127, '8fc902'},
	PairU{19257, 'b99601'},
	PairU{17684, '948a01'},
	PairU{1325, 'ad0a'},
	PairU{61840, '90e303'},
	PairU{40710, '86be02'},
	PairU{62082, '82e503'},
	PairU{23342, 'aeb601'},
	PairU{35058, 'f29102'},
	PairU{40159, 'dfb902'},
	PairU{44189, '9dd902'},
	PairU{62036, 'd4e403'},
	PairU{41247, '9fc202'},
	PairU{59480, 'd8d003'},
	PairU{38184, 'a8aa02'},
	PairU{15638, '967a'},
	PairU{59441, 'b1d003'},
	PairU{35077, '859202'},
	PairU{3723, '8b1d'},
	PairU{19303, 'e79601'},
	PairU{3842, '821e'},
	PairU{26673, 'b1d001'},
	PairU{65376, 'e0fe03'},
	PairU{41853, 'fdc602'},
	PairU{29944, 'f8e901'},
	PairU{37899, '8ba802'},
	PairU{58497, '81c903'},
	PairU{19096, '989501'},
	PairU{15674, 'ba7a'},
	PairU{28569, '99df01'},
	PairU{64493, 'edf703'},
	PairU{32436, 'b4fd01'},
	PairU{6647, 'f733'},
	PairU{45867, 'abe602'},
	PairU{15436, 'cc78'},
	PairU{50918, 'e68d03'},
	PairU{14744, '9873'},
	PairU{35229, '9d9302'},
	PairU{24528, 'd0bf01'},
	PairU{50456, '988a03'},
	PairU{38948, 'a4b002'},
	PairU{44381, 'ddda02'},
	PairU{44414, 'feda02'},
	PairU{53514, '8aa203'},
	PairU{26661, 'a5d001'},
	PairU{6936, '9836'},
	PairU{64439, 'b7f703'},
	PairU{6416, '9032'},
	PairU{64900, '84fb03'},
	PairU{16493, 'ed8001'},
	PairU{60566, '96d903'},
	PairU{43060, 'b4d002'},
	PairU{46111, '9fe802'},
	PairU{11477, 'd559'},
	PairU{3309, 'ed19'},
	PairU{45345, 'a1e202'},
	PairU{42714, 'dacd02'},
	PairU{53006, '8e9e03'},
	PairU{4101, '8520'},
	PairU{24645, 'c5c001'},
	PairU{5723, 'db2c'},
	PairU{47501, '8df302'},
	PairU{57066, 'eabd03'},
	PairU{17033, '898501'},
	PairU{6346, 'ca31'},
	PairU{12568, '9862'},
	PairU{62223, '8fe603'},
	PairU{56084, '94b603'},
	PairU{56131, 'c3b603'},
	PairU{29969, '91ea01'},
	PairU{32369, 'f1fc01'},
	PairU{61495, 'b7e003'},
	PairU{44949, '95df02'},
	PairU{50004, 'd48603'},
	PairU{9220, '8448'},
	PairU{10256, '9050'},
	PairU{45732, 'a4e502'},
	PairU{43942, 'a6d702'},
	PairU{1469, 'bd0b'},
	PairU{28982, 'b6e201'},
	PairU{417, 'a103'},
	PairU{58457, 'd9c803'},
	PairU{37589, 'd5a502'},
	PairU{29383, 'c7e501'},
	PairU{21056, 'c0a401'},
	PairU{4745, '8925'},
	PairU{35514, 'ba9502'},
	PairU{63646, '9ef103'},
	PairU{2842, '9a16'},
	PairU{21858, 'e2aa01'},
	PairU{40388, 'c4bb02'},
	PairU{30328, 'f8ec01'},
	PairU{39951, '8fb802'},
	PairU{23997, 'bdbb01'},
	PairU{62946, 'e2eb03'},
	PairU{35650, 'c29602'},
	PairU{8750, 'ae44'},
	PairU{46337, '81ea02'},
	PairU{22273, '81ae01'},
	PairU{53644, '8ca303'},
	PairU{11718, 'c65b'},
	PairU{3178, 'ea18'},
	PairU{33674, '8a8702'},
	PairU{30684, 'dcef01'},
	PairU{2316, '8c12'},
	PairU{58965, 'd5cc03'},
	PairU{33215, 'bf8302'},
	PairU{40078, '8eb902'},
	PairU{27427, 'a3d601'},
	PairU{32957, 'bd8102'},
	PairU{16508, 'fc8001'},
	PairU{39346, 'b2b302'},
	PairU{29215, '9fe401'},
	PairU{4932, 'c426'},
	PairU{58679, 'b7ca03'},
	PairU{64203, 'cbf503'},
	PairU{30720, '80f001'},
	PairU{55219, 'b3af03'},
	PairU{13026, 'e265'},
	PairU{18918, 'e69301'},
	PairU{55193, '99af03'},
	PairU{54315, 'aba803'},
	PairU{39709, '9db602'},
	PairU{11413, '9559'},
	PairU{7492, 'c43a'},
	PairU{31907, 'a3f901'},
	PairU{7409, 'f139'},
	PairU{33806, '8e8802'},
	PairU{13508, 'c469'},
	PairU{56451, '83b903'},
	PairU{35356, '9c9402'},
	PairU{55376, 'd0b003'},
	PairU{17814, '968b01'},
	PairU{64517, '85f803'},
	PairU{2994, 'b217'},
	PairU{59169, 'a1ce03'},
	PairU{56207, '8fb703'},
	PairU{22147, '83ad01'},
	PairU{14012, 'bc6d'},
	PairU{13054, 'fe65'},
	PairU{4892, '9c26'},
	PairU{42371, '83cb02'},
	PairU{8325, '8541'},
	PairU{54057, 'a9a603'},
	PairU{45378, 'c2e202'},
	PairU{43300, 'a4d202'},
	PairU{32715, 'cbff01'},
	PairU{39871, 'bfb702'},
	PairU{55492, 'c4b103'},
	PairU{58680, 'b8ca03'},
	PairU{33359, 'cf8402'},
	PairU{34042, 'fa8902'},
	PairU{33938, '928902'},
	PairU{2765, 'cd15'},
	PairU{19810, 'e29a01'},
	PairU{949, 'b507'},
	PairU{61490, 'b2e003'},
	PairU{57261, 'adbf03'},
	PairU{63475, 'f3ef03'},
	PairU{21287, 'a7a601'},
	PairU{47662, 'aef402'},
	PairU{56098, 'a2b603'},
	PairU{54230, 'd6a703'},
	PairU{39149, 'edb102'},
	PairU{45100, 'ace002'},
	PairU{60998, 'c6dc03'},
	PairU{31905, 'a1f901'},
	PairU{22804, '94b201'},
	PairU{44510, 'dedb02'},
	PairU{15248, '9077'},
	PairU{14713, 'f972'},
	PairU{58289, 'b1c703'},
	PairU{8585, '8943'},
	PairU{43338, 'cad202'},
	PairU{34636, 'cc8e02'},
	PairU{18370, 'c28f01'},
	PairU{31070, 'def201'},
	PairU{33749, 'd58702'},
	PairU{47545, 'b9f302'},
	PairU{2995, 'b317'},
	PairU{35572, 'f49502'},
	PairU{56226, 'a2b703'},
	PairU{37064, 'c8a102'},
	PairU{65522, 'f2ff03'},
	PairU{32768, '808002'},
	PairU{25070, 'eec301'},
	PairU{20130, 'a29d01'},
	PairU{64278, '96f603'},
	PairU{37498, 'faa402'},
	PairU{40782, 'cebe02'},
	PairU{59649, '81d203'},
	PairU{60135, 'e7d503'},
	PairU{4690, 'd224'},
	PairU{4000, 'a01f'},
	PairU{54743, 'd7ab03'},
	PairU{33671, '878702'},
	PairU{1576622459, 'fbb2e5ef05'},
	PairU{2285263560, 'c8b5d9c108'},
	PairU{263450205, 'dddccf7d'},
	PairU{2633373322, '8aadd8e709'},
	PairU{749077709, 'cd8998e502'},
	PairU{813712125, 'fd85818403'},
	PairU{2733002807, 'b7a099970a'},
	PairU{2296034148, 'e4e6eac608'},
	PairU{4189355970, 'c2ffd1cd0f'},
	PairU{2204900736, '80bbb09b08'},
	PairU{509345770, 'eaffeff201'},
	PairU{3884454891, 'eba7a0bc0e'},
	PairU{3537905033, '89cb80970d'},
	PairU{2839422414, 'cecbf8c90a'},
	PairU{2428204406, 'f6eaed8509'},
	PairU{3728526525, 'bd99f3f10d'},
	PairU{3049593858, '82b894ae0b'},
	PairU{1457544425, 'e9b981b705'},
	PairU{1469203333, '8587c9bc05'},
	PairU{3774511384, '98f2e9870e'},
	PairU{2787404081, 'b1d291b10a'},
	PairU{3996089698, 'e2fabdf10e'},
	PairU{2382104382, 'be8ef0ef08'},
	PairU{2936137684, 'd4cf87f80a'},
	PairU{3779422746, '9ad4958a0e'},
	PairU{4245632140, '8ce9bce80f'},
	PairU{2172962693, '858f938c08'},
	PairU{1599194849, 'e18dc7fa05'},
	PairU{719183242, '8abbf7d602'},
	PairU{632925525, 'd5dae6ad02'},
	PairU{4083247540, 'b4d3859b0f'},
	PairU{499253479, 'e78188ee01'},
	PairU{3010530387, 'd398c49b0b'},
	PairU{707821343, '9ffec1d102'},
	PairU{875381905, '9189b5a103'},
	PairU{4219956182, 'd6d79ddc0f'},
	PairU{1550196940, 'ccc198e305'},
	PairU{830853841, 'd1a5978c03'},
	PairU{1683329584, 'b0a4d6a206'},
	PairU{2589390623, '9feedbd209'},
	PairU{383008357, 'e5fcd0b601'},
	PairU{2197647374, '8ee0f59708'},
	PairU{3005775514, '9afda1990b'},
	PairU{4273640298, 'eaa6eaf50f'},
	PairU{3148225718, 'b6b998dd0b'},
	PairU{3349023016, 'a892f8bc0c'},
	PairU{3347759669, 'b584abbc0c'},
	PairU{1039207051, '8b95c4ef03'},
	PairU{1139110863, 'cfe7959f04'},
	PairU{2745283454, 'fee6869d0a'},
	PairU{732240829, 'bdb794dd02'},
	PairU{338637064, '88e2bca101'},
	PairU{837811858, '92fdbf8f03'},
	PairU{1899187297, 'e198cd8907'},
	PairU{1793703115, 'cbf9a6d706'},
	PairU{2581410360, 'b8e4f4ce09'},
	PairU{3137095172, '848cf1d70b'},
	PairU{7424679, 'a795c503'},
	PairU{1449454191, 'efd493b305'},
	PairU{4079856829, 'bdd9b6990f'},
	PairU{2571092296, 'c882ffc909'},
	PairU{3717759897, '9987e2ec0d'},
	PairU{2883096695, 'f7a0e2de0a'},
	PairU{1691131815, 'a7bfb2a606'},
	PairU{2129105334, 'b6a39ef707'},
	PairU{1009980302, '8ea7cce103'},
	PairU{3434162685, 'fdd3c4e50c'},
	PairU{639957245, 'fdf193b102'},
	PairU{932838607, 'cff9e7bc03'},
	PairU{3235504188, 'bcc0e7860c'},
	PairU{1180182155, '8bcde0b204'},
	PairU{2995829608, 'e8f6c2940b'},
	PairU{965674411, 'ab8bbccc03'},
	PairU{378543011, 'a3b7c0b401'},
	PairU{4285571791, 'cfc5c2fb0f'},
	PairU{282020545, 'c195bd8601'},
	PairU{2405125211, 'db98edfa08'},
	PairU{869729327, 'af88dc9e03'},
	PairU{2165033856, '8097af8808'},
	PairU{3190934636, 'ec98c7f10b'},
	PairU{1560913891, 'e3cfa6e805'},
	PairU{3990487004, 'dcffe7ee0e'},
	PairU{3183785705, 'e9ed92ee0b'},
	PairU{2461047003, 'dbb1c29509'},
	PairU{4262615991, 'b7b7c9f00f'},
	PairU{1414423930, 'facab9a205'},
	PairU{999371334, 'c6e4c4dc03'},
	PairU{1689190329, 'b9ffbba506'},
	PairU{68235555, 'a3e2c420'},
	PairU{248856297, 'e9fdd476'},
	PairU{4092196862, 'feefa79f0f'},
	PairU{3494374883, 'e3db9f820d'},
	PairU{2813755235, 'e3fed9bd0a'},
	PairU{1313825099, 'cbc2bdf204'},
	PairU{2591128222, '9ef5c5d309'},
	PairU{1523966010, 'bac0d7d605'},
	PairU{1127625082, 'fae2d89904'},
	PairU{1256675466, '8ab19dd704'},
	PairU{1190518037, '95bad7b704'},
	PairU{2791367114, 'cac383b30a'},
	PairU{3774432974, 'ce8de5870e'},
	PairU{3414245763, '838385dc0c'},
	PairU{1597034808, 'b8a2c3f905'},
	PairU{4007859466, '8aaa8cf70e'},
	PairU{2581759998, 'fe8f8acf09'},
	PairU{3207291398, '86c4adf90b'},
	PairU{2641580787, 'f3a5cdeb09'},
	PairU{4203404220, 'bcb7abd40f'},
	PairU{3827909534, '9e87a5a10e'},
	PairU{3400985995, '8bdbdbd50c'},
	PairU{2567620135, 'a78cabc809'},
	PairU{2308950307, 'a392ffcc08'},
	PairU{1314838143, 'ffacfbf204'},
	PairU{2178914725, 'a5b3fe8e08'},
	PairU{1733225139, 'b3d5bbba06'},
	PairU{1553515984, 'd08be3e405'},
	PairU{2137829658, '9ae2b2fb07'},
	PairU{4142098808, 'f8d28db70f'},
	PairU{4103958027, '8bdcf5a40f'},
	PairU{2206666819, 'c3a09c9c08'},
	PairU{1716906204, 'dcd1d7b206'},
	PairU{2494558124, 'acdfbfa509'},
	PairU{1986252720, 'b09f8fb307'},
	PairU{2649315202, '82afa5ef09'},
	PairU{3175882084, 'e4bab0ea0b'},
	PairU{3749882228, 'f4d28afc0d'},
	PairU{3674100778, 'aaa8f9d70d'},
	PairU{1053226166, 'b6e99bf603'},
	PairU{153954905, 'd9d4b449'},
	PairU{1870883730, '92d78dfc06'},
	PairU{2505819863, 'd78defaa09'},
	PairU{2401253919, '9ff480f908'},
	PairU{340593328, 'b095b4a201'},
	PairU{969765839, 'cfe7b5ce03'},
	PairU{1456639920, 'b09fcab605'},
	PairU{91177824, 'e086bd2b'},
	PairU{3495457228, 'cce3e1820d'},
	PairU{3212735579, 'dbe8f9fb0b'},
	PairU{275512216, '98f7af8301'},
	PairU{672284862, 'be81c9c002'},
	PairU{961912600, '98bed6ca03'},
	PairU{3084835320, 'f8b3fbbe0b'},
	PairU{220020303, 'cffcf468'},
	PairU{2450163298, 'e28caa9009'},
	PairU{4055401811, 'd38ae28d0f'},
	PairU{1163553324, 'acd4e9aa04'},
	PairU{1202551545, 'f9f5b5bd04'},
	PairU{2019542236, 'dc89ffc207'},
	PairU{2761376567, 'b786dda40a'},
	PairU{1609605426, 'b2c2c2ff05'},
	PairU{516942760, 'a8d7bff601'},
	PairU{2377899272, '88baefed08'},
	PairU{3580860466, 'b2b0beab0d'},
	PairU{2126356025, 'b9bcf6f507'},
	PairU{1595762393, 'd9cdf5f805'},
	PairU{2883179007, 'ffa3e7de0a'},
	PairU{3535907129, 'b9d286960d'},
	PairU{1158802511, 'cfd8c7a804'},
	PairU{497717516, '8ca2aaed01'},
	PairU{2629767619, 'c3a3fce509'},
	PairU{75674906, '9aea8a24'},
	PairU{710236774, 'e6b4d5d202'},
	PairU{2645675237, 'e599c7ed09'},
	PairU{2817817952, 'e0fad1bf0a'},
	PairU{1074553497, '99c5b18004'},
	PairU{742113873, 'd184efe102'},
	PairU{2387606507, 'ebf7bff208'},
	PairU{3826295571, '93c6c2a00e'},
	PairU{1474946464, 'a0cba7bf05'},
	PairU{3475754455, 'd79baff90c'},
	PairU{4126313733, '859acaaf0f'},
	PairU{1661751507, 'd3a1b19806'},
	PairU{3069717007, '8fd4e0b70b'},
	PairU{3337222955, 'abf6a7b70c'},
	PairU{1706153109, '95a9c7ad06'},
	PairU{1402000705, 'c1aac39c05'},
	PairU{2251211916, '8c89bbb108'},
	PairU{2798865857, 'c19bcdb60a'},
	PairU{1641244705, 'a1d0cd8e06'},
	PairU{2512858203, 'dbd89cae09'},
	PairU{524622237, '9db394fa01'},
	PairU{3771544098, 'a2e4b4860e'},
	PairU{4107691575, 'b7ccd9a60f'},
	PairU{603936895, 'ffb0fd9f02'},
	PairU{3344176246, 'f6a8d0ba0c'},
	PairU{1183276330, 'aaba9db404'},
	PairU{3597650236, 'bc92bfb30d'},
	PairU{441168468, 'd4e4aed201'},
	PairU{4221874260, 'd4e092dd0f'},
	PairU{3950335172, 'c4a9d5db0e'},
	PairU{1973360424, 'a8aefcac07'},
	PairU{3823824438, 'b6dcab9f0e'},
	PairU{2049800081, '91efb5d107'},
	PairU{3660178372, 'c4c7a7d10d'},
	PairU{780774180, 'a4d6a6f402'},
	PairU{2681864530, 'd282e8fe09'},
	PairU{1528412212, 'b4f0e6d805'},
	PairU{2376378142, '9ece92ed08'},
	PairU{1064623603, 'f3bbd3fb03'},
	PairU{3233206445, 'ada1db850c'},
	PairU{3672796657, 'f1dba9d70d'},
	PairU{781087442, 'd2e5b9f402'},
	PairU{2245394147, 'e3fdd7ae08'},
	PairU{3836660377, '9995bba50e'},
	PairU{2353965320, '88d2bae208'},
	PairU{4272222672, 'd0e393f50f'},
	PairU{4111629699, '83fbc9a80f'},
	PairU{913118848, '80adb4b303'},
	PairU{185527771, 'dbdbbb58'},
	PairU{4059618015, 'dfb5e38f0f'},
	PairU{790485926, 'a6b7f7f802'},
	PairU{3884507512, 'f8c2a3bc0e'},
	PairU{3994954592, 'e0d6f8f00e'},
	PairU{331228009, 'e9c6f89d01'},
	PairU{1805517085, '9d82f8dc06'},
	PairU{14100549, 'c5d0dc06'},
	PairU{4142777074, 'f285b7b70f'},
	PairU{3788467130, 'bad7bd8e0e'},
	PairU{2905111067, '9bf4a1e90a'},
	PairU{1531557470, 'deeca6da05'},
	PairU{806936703, 'ffc0e38003'},
	PairU{805343149, 'ad9f828003'},
	PairU{922694503, 'e7e6fcb703'},
	PairU{3739660092, 'bcde9af70d'},
	PairU{2976345039, 'cfd79d8b0b'},
	PairU{1149278908, 'bcb582a404'},
	PairU{2399355301, 'a5838df808'},
	PairU{2444829847, '97c9e48d09'},
	PairU{2986169099, '8ba6f58f0b'},
	PairU{2950144350, 'dec2defe0a'},
	PairU{3916312506, 'badfb8cb0e'},
	PairU{845785612, '8cd4a69303'},
	PairU{297510487, 'd7ccee8d01'},
	PairU{3841161226, '8af0cda70e'},
	PairU{69098825, 'c9baf920'},
	PairU{3026902730, 'cabdaba30b'},
	PairU{1453144899, 'c3f6f4b405'},
	PairU{2503664640, '80c8eba909'},
	PairU{1248214829, 'adfe98d304'},
	PairU{1441945961, 'e9b2c9af05'},
	PairU{800833378, 'e2feeefd02'},
	PairU{1050278887, 'e7f7e7f403'},
	PairU{4033383511, 'd798a2830f'},
	PairU{4273706408, 'a8abeef50f'},
	PairU{2297733394, '92c2d2c708'},
	PairU{2870451344, '90b9ded80a'},
	PairU{4039087780, 'a4adfe850f'},
	PairU{693267104, 'a0d5c9ca02'},
	PairU{539765297, 'b1d4b08102'},
	PairU{4047558201, 'b9ac838a0f'},
	PairU{2933694237, '9dbef2f60a'},
	PairU{3918895609, 'f9b3d6cc0e'},
	PairU{4250667884, 'ec96f0ea0f'},
	PairU{2521021176, 'f8f58eb209'},
	PairU{3749978539, 'abc390fc0d'},
	PairU{4258793638, 'a691e0ee0f'},
	PairU{16497666429405569624, 'd8c4ecdcd886dff9e401'},
	PairU{14536154376438558816, 'e0e0f0bebbecb2ddc901'},
	PairU{6843499419089188695, 'd7ee9cbbe9cebffc5e'},
	PairU{7841396255184312349, '9db0dbea938b8fe96c'},
	PairU{1113684928626427234, 'e28af093acd1a6ba0f'},
	PairU{4134256720751534067, 'f3bfaa95d592f5af39'},
	PairU{17253201377649926326, 'b6c1eac8dde7ebb7ef01'},
	PairU{15469952299346770796, 'ecf6b4b6f2f493d8d601'},
	PairU{5835458901455749877, 'f585a5d5c0deedfd50'},
	PairU{17844543005813556250, '9a88cb859caaa3d2f701'},
	PairU{4969364902516834905, 'd984bdeecce1aefb44'},
	PairU{6692377965772467354, '9ad993c4d5c986f05c'},
	PairU{1475587601107919743, 'ffc688f8aee395bd14'},
	PairU{3843212937304183130, 'daa2cea1deb9f5aa35'},
	PairU{9777013101649715726, '8ec4afc1d7b3bbd78701'},
	PairU{1570152583066146974, '9eb1c7e6c8ae93e515'},
	PairU{17356087171625683816, 'e896dadbb7aacdeef001'},
	PairU{5056090879529063593, 'a9a9b0d3ddfbb59546'},
	PairU{3384806410955606790, '8696e5f3f4efcffc2e'},
	PairU{7442628686330531987, '93a1f49eb8ebe1a467'},
	PairU{1357168659508766305, 'e1ccc0d2c8b6e8ea12'},
	PairU{7074114660671564637, 'ddd6d8cddbb9939662'},
	PairU{16717941301969772540, 'fcbff5bd9ee18381e801'},
	PairU{2387629661863056173, 'ada6a2d4ef90a49121'},
	PairU{7862375172691359479, 'f7edf9d88192b18e6d'},
	PairU{17677645154584602864, 'f0d1ae83ba93e7a9f501'},
	PairU{4045118881321648135, '87809cdb8cc6c99138'},
	PairU{9095902393560545338, 'baf0b4be8ee1c89d7e'},
	PairU{1068136536003572080, 'f0e2ef80d390b2e90e'},
	PairU{7337278415349509923, 'a3deb1e881fbcfe965'},
	PairU{1740551343930355483, '9bdeb0c29bc7eb9318'},
	PairU{3893609823214902851, 'c3bcc8b594b0b88436'},
	PairU{16732013103060218250, '8ae3d7c8c6a8839ae801'},
	PairU{10226525623380170998, 'f6b185e0f3dbfaf58d01'},
	PairU{1895133965637192732, '9c98d4bcd0c9b7a61a'},
	PairU{15948109888341212272, 'f0f8e3eba5aec4a9dd01'},
	PairU{8092118975093648323, 'c3d79ef5d7eabea670'},
	PairU{10514734800163341226, 'aa87b2f9f5f2f5f59101'},
	PairU{13884858018854964648, 'a89bd694d999bbd8c001'},
	PairU{5506039178247142939, '9bfcb7d6ddaed8b44c'},
	PairU{3841378177801331069, 'fdc2a4be9aa3d4a735'},
	PairU{17760608059192644565, 'd5a7f8d8e1dd96bdf601'},
	PairU{7019050392006902605, 'cd868de1c9a4abb461'},
	PairU{8327374118332948680, 'c8c191d8bed4b1c873'},
	PairU{2563897145003893317, 'c5c4e1c4e8dbb2ca23'},
	PairU{12643216874244515033, 'd9c9c6fcaad8eebaaf01'},
	PairU{13119112646413862286, '8e8bd58e9aee9c88b601'},
	PairU{18108784008520238429, 'dd8afef0c7e3d4a7fb01'},
	PairU{1873271405493943285, 'f5f7f3a2b1cdccff19'},
	PairU{8874913961157715656, 'c8cdd7979fe781957b'},
	PairU{12117614803696436543, 'bf8adddae4ce9a95a801'},
	PairU{13622021734351412017, 'b1b6aab58493c985bd01'},
	PairU{11227690731560178118, 'c6a3dcb4bca7b1e89b01'},
	PairU{15075290977826340765, '9dfff597daa88c9bd101'},
	PairU{3782283879983012434, 'd2b4cb90e4e4d7be34'},
	PairU{2286873011306353768, 'e8a8dfeab2dca6de1f'},
	PairU{14474524103216191221, 'f5ad9fc8b8dff5efc801'},
	PairU{10516557500424622678, 'd6a49ec6bdaa94f99101'},
	PairU{6271761304563999046, 'c6c2cf94d6b5f18457'},
	PairU{7003418245793840538, '9afb9f909ff9c89861'},
	PairU{4738544158561718530, '82a2cbdce699ace141'},
	PairU{3144190332643264855, 'd792addbae8e9ad12b'},
	PairU{227823842728710, '86b6def9c5e633'},
	PairU{10405418797642988611, 'c3f8baa5c2a8deb39001'},
	PairU{8444002472442420534, 'b6bacde3aaf0c79775'},
	PairU{3878245798820074584, 'd8a8c9b2998093e935'},
	PairU{8012357224138885541, 'a593d5f29a8ee7986f'},
	PairU{11962454006436529282, '8289e4fc9f91cb81a601'},
	PairU{12665910134576977208, 'b8aaecc391c596e3af01'},
	PairU{15068877586868647463, 'a7ec9a89bb8ada8fd101'},
	PairU{4210079315543515777, '81c5c3f9cd9acdb63a'},
	PairU{11487841127191890497, 'c19495b3bdd4c0b69f01'},
	PairU{10485683442469938049, '81c7f6fc90b1a8c29101'},
	PairU{11204063524666424848, '90fc9ce59f8db5be9b01'},
	PairU{5415006522766295924, 'f486a2afe1f7fd924b'},
	PairU{4630711960513531005, 'fd98bfbce5ffe5a140'},
	PairU{3982128255237019230, 'deaccbbbcc91d7a137'},
	PairU{6617393968943300669, 'bd80b6cd9198edea5b'},
	PairU{1435843736633158116, 'e4d3f497d688c9f613'},
	PairU{14187365854017477836, 'ccb9dfd9b8c2e9f1c401'},
	PairU{8735249437125091689, 'e9c296e2cee2f59c79'},
	PairU{6376283519002828267, 'ebbbddcabd82c7be58'},
	PairU{3657563100074945951, '9fdbacdcacc891e132'},
	PairU{4437625949942147625, 'a9d4edc2a3a9e7ca3d'},
	PairU{1885997661622250280, 'a8d6d8e1979c9a961a'},
	PairU{16732215193639573532, '9cb09fd294a2b19ae801'},
	PairU{5667120534244932029, 'bdf3f6b79a83ead24e'},
	PairU{11825700855638588116, 'd4f5fb82de88d58ea401'},
	PairU{4527576189882154517, '95fcc4dbe7d1cbea3e'},
	PairU{1280682687926913479, 'c79381dfe8c3f9e211'},
	PairU{9740039616846814260, 'b4b88ba3d3cde4958701'},
	PairU{3956728337499169408, '80c5f0aad9eec7f436'},
	PairU{17343838944915026409, 'e9e3fbc98fb4ecd8f001'},
	PairU{16687025645117266970, '9ab8b1f899ad8ecae701'},
	PairU{15670949158519500554, '8ade8bf6b3a799bdd901'},
	PairU{16466066235311737763, 'a397bd9e8080cec1e401'},
	PairU{15451913260127887220, 'f4bec8d0e0a78eb8d601'},
	PairU{9278357653237088389, '8581e5a187a4d6e18001'},
	PairU{1075878753333237396, '94adc58eeac092f70e'},
	PairU{3563137885558970298, 'baafea93f5e0b3b931'},
	PairU{8863797907167105568, 'a0a4af8cbfa7a2817b'},
	PairU{734695070650939283, '93ef8483e3a88a990a'},
	PairU{8002043003097211607, 'd7c589a8eff6bd866f'},
	PairU{5192771009703340673, '81d5c1ca88b79b8848'},
	PairU{13179126492408335310, 'ce97c1af80b7eaf2b601'},
	PairU{17790454869050497143, 'f7b0fef2a08e99f2f601'},
	PairU{7616993389808968746, 'aac8bfc09ae4bfda69'},
	PairU{16318616824692571020, '8cf78dacadf1d7bbe201'},
	PairU{8765939449468784786, '92a1ae81c4efb7d379'},
	PairU{75261474164892201, 'a9c4fbc697bdd88501'},
	PairU{3133664910589528302, 'ee91ae86a2f4c0be2b'},
	PairU{15097000880341691964, 'bcdc9ab483cad4c1d101'},
	PairU{3022973414962426465, 'e1e4a3a2dfc9f0f929'},
	PairU{5609378649501280170, 'aacfbcb19785a1ec4d'},
	PairU{990738819831151138, 'a2f4e6e9d1f6f3df0d'},
	PairU{12348568270978276187, 'dbb6c7e2abadbbafab01'},
	PairU{16901939111747112304, 'f09adf97a682f0c7ea01'},
	PairU{9801441920966166937, '9993be8ef3efed828801'},
	PairU{10060108146541440626, 'f2dcf6e4efe1abce8b01'},
	PairU{11316140797099052479, 'bf9b94bb97c3c0859d01'},
	PairU{1236259003500982012, 'fcddb790b8e0849411'},
	PairU{16823652378989721398, 'b6eed992c0d6e7bce901'},
	PairU{15529439428856688579, 'c3dfcaef9cdce9c1d701'},
	PairU{10242529200753152505, 'f9a3d4a3a7c1b1928e01'},
	PairU{5245335121816452570, 'dafbffbd8990cbe548'},
	PairU{2747545226467382795, '8baca2dfb8bacf9026'},
	PairU{13609900213639311091, 'f3ddeff3ac8485f0bc01'},
	PairU{1874387577747180441, '99979af8a2b2ca811a'},
	PairU{10272916768788366883, 'a38ca3d6f7ebaec88e01'},
	PairU{1114278600321221100, 'ec83ddc7bbcfadbb0f'},
	PairU{9897207218105991608, 'b8838fbdefb0fcac8901'},
	PairU{8846810664494540287, 'ffcbd6e2d4ed8be37a'},
	PairU{10562228117401896415, 'df8b88f3afd0a4ca9201'},
	PairU{8608031914317356491, 'cbbbf2a0b3edf7ba77'},
	PairU{1865654390810688288, 'a0f6acaf8ad988f219'},
	PairU{12094674477893877376, '80fde3c6b7cbbaeca701'},
	PairU{8558966579069034438, 'c6bfe6ee99d8e3e376'},
	PairU{5203591044982965645, '8d8bf484c5cfb79b48'},
	PairU{11233273182042669342, '9e82cd8495cea6f29b01'},
	PairU{16017135281365759761, '91eebae9d0f592a4de01'},
	PairU{6945363918775826385, 'd1efcaf3f9f5b8b160'},
	PairU{6045420329007298572, '8cf09ee3a5b8e9f253'},
	PairU{5510634334482130396, 'dccb83cfb097edbc4c'},
	PairU{17468619073700264256, 'c0f291e4ea8fc0b6f201'},
	PairU{18247503505561409945, '99b3bbb5a4f8899efd01'},
	PairU{1988695039543172212, 'f4b8b5b4a1f3d0cc1b'},
	PairU{13715042894693961232, '909ceacb8edbe7aabe01'},
	PairU{2018975808039210530, 'a29ca3c2cef9b5821c'},
	PairU{15332238884456094609, '91b7bffb81c1c3e3d401'},
	PairU{7652912018846076278, 'f69a8de5f2dda69a6a'},
	PairU{15768132723588903218, 'b2e2ea85b4a5eae9da01'},
	PairU{17375028618161701841, 'd18799de8a8fa090f101'},
	PairU{10110759008783538163, 'f3affec0fdb7a8a88c01'},
	PairU{3415009868491440130, '82808b9f9caca3b22f'},
	PairU{15561213538269371436, 'acb8aeb4c2a8a2fad701'},
	PairU{15315359322966216613, 'a59ff9a68fc5c5c5d401'},
	PairU{2458535091929179151, '8fc895b2be949e8f22'},
	PairU{17358307530478456555, 'eb9da081b197c6f2f001'},
	PairU{7363375970986704126, 'fef9bcd2ecedfd9766'},
	PairU{14738538189932390152, '88cecbbbd4ccf3c4cc01'},
	PairU{14440442733995446828, 'aca4adb986c5b0b3c801'},
	PairU{1393366409401905860, 'c4e5f292dfeb8eab13'},
	PairU{1900119664317415511, 'd780ffb2c798a5af1a'},
	PairU{14109693877532310998, 'd6fbc49eb4faece7c301'},
	PairU{14769995374197693954, '82bcdde99b91e4fccc01'},
	PairU{4999940398370050992, 'b0a7e3f7d2e9d6b145'},
	PairU{5047528203842621028, 'e49cb8fe88859b8646'},
	PairU{7485339658625876641, 'a185a8fcaa98d1f067'},
	PairU{11439500404765512453, '85a6f18fa5a0d1e09e01'},
	PairU{8070986252393944622, 'aeac8288c1e7f98070'},
	PairU{1050338103191410594, 'a2afa0f6899ee3c90e'},
	PairU{12530379333244790682, '9acfce97d6b3b6f2ad01'},
	PairU{1427964220740374637, 'eda0ca89cabcc9e813'},
	PairU{6082603178276280450, '82b1c7b6d1ebefb454'},
	PairU{5205595947569509765, '85ebdf98f1bdff9e48'},
	PairU{8731691902328862096, '90ebddf3dcf0cc9679'},
	PairU{3751617101016806029, '8de58cfb85fc9a8834'},
	PairU{14299237009190957872, 'b0e6c8dfc989c6b8c601'},
	PairU{13867437321366946997, 'b589c9e8d598c2b9c001'},
	PairU{11573419706116713613, '8dd9de9cf7fcc2cea001'},
	PairU{2191009178272376218, '9acbaed4d3e781b41e'},
	PairU{10715779747999238730, 'ca94c7d6ff9c86db9401'},
	PairU{10469894606496237985, 'a193f2bfc3b5a2a69101'},
	PairU{14159978080673443788, 'ccdfeeaeaaa196c1c401'},
	PairU{13587730528178406622, 'de89b8deca9dd4c8bc01'},
	PairU{11257190893642734044, 'dcb3d98298efe49c9c01'},
	PairU{2928980676484332701, '9df9d3be828df5d228'},
	PairU{16488195901431596315, '9b8a87df86daf5e8e401'},
	PairU{4278215864147998546, 'd2eeedb795d5d1af3b'},
	PairU{4323175936829160300, 'ecf6df81c0b3c0ff3b'},
	PairU{13076173724363569287, '87f1f2af8cd7f9bbb501'},
	PairU{8717063203043061758, 'fedf8ab0c5d9cefc78'},
	PairU{5138682769057202779, 'db84cdb88a9891a847'},
	PairU{13344414739953353340, 'fc84a0b391d0b898b901'},
	PairU{11076654462981947196, 'bcfea2aec2d18bdc9901'},
	PairU{8460080481274843743, 'dfa4e98cfdcb8fb475'},
	PairU{13685618867296918967, 'b7bbf7e49abbc5f6bd01'},
	PairU{8053099365893889619, 'd3bc9da8cce696e16f'},
	PairU{10532633304015060167, 'c7f990daf8c5db959201'},
	PairU{11112029803825057860, 'c4a0afb4b887f79a9a01'},
	PairU{8069181957144908666, 'faa6f4bccdc7dffd6f'},
	PairU{8204921825036634609, 'f193a2b9da9defee71'},
	PairU{3700587972716690644, 'd4f9f1aef9a4c8ad33'},
	PairU{9488398371130531571, 'f3bd85f5bf81e4d68301'},
	PairU{1406191118217599678, 'beb5adbdf2ebf2c113'},
	PairU{14495297497968030852, '84e9f8bfe688e994c901'},
	PairU{7593087050568726701, 'adf191ee948e84b069'},
	PairU{13679751921386162979, 'a396c28dcebc8fecbd01'},
	PairU{8244731277729298875, 'bbf3edaca2edcab572'},
	PairU{4444393973837935956, 'd4eafca3d998ead63d'},
	PairU{4043542914206254024, 'c887bfdab59be38e38'},
	PairU{10186751003295074837, '959487b68d82a7af8d01'},
	PairU{15777457821124589455, '8fefdbb6bbc9b2fada01'},
	PairU{10105846657260489569, 'e196d284debfcb9f8c01'},
	PairU{717755936680095145, 'a9dbfac98be7fefa09'},
	PairU{17680852272158274921, 'e9ea9b81eeaec0aff501'},
	PairU{5528732412273215974, 'e693ef9de29a80dd4c'},
	PairU{15989613841619609519, 'afcf9faba8a2a1f3dd01'},
	PairU{5511632168482070686, '9ec1f5f79588d0be4c'},
	PairU{5281614765525499435, 'abfceb9cd69484a649'},
	PairU{8127488305158606999, '97a1fb88d6f1a8e570'},
	PairU{10607748610367498417, 'b1d1e4dc8ae5929b9301'},
	PairU{6827839821837000505, 'b9bef1dec784d7e05e'},
	PairU{1174735752806106832, 'd08dd69797fedfa610'},
	PairU{1152991407683701572, 'c496c2a0b9f28f8010'},
	PairU{168485408808666162, 'b28882efe292a5ab02'},
	PairU{1500959912189480795, 'db9e9fa1e7e29eea14'},
	PairU{16925540598690529784, 'f8839bc3fcafe6f1ea01'},
	PairU{5788166731354978227, 'b3ef96999adfeca950'},
	PairU{17431587162243082439, 'c7d188a6ae85dcf4f101'},
	PairU{3298539645723045748, 'f4def5a6cd8ab1e32d'},
	PairU{2944295686823961901, 'adcaedf6bdaa8fee28'},
	PairU{5601897801686080844, 'ccbaa9cced8bfcde4d'},
	PairU{4294505698158362143, '9ffcfa84dec5c9cc3b'},
	PairU{2790287985116613059, 'c393aaefb884c6dc26'},
	PairU{13025394675902087618, 'c283a2e3a3eedfe1b401'},
	PairU{3449890764261488679, 'a7f0d3edf2ab9ef02f'},
	PairU{9345651036153697948, '9cadcfe5a1839bd98101'},
	PairU{11217857190501866486, 'f6b7b3a7e1b5f5d69b01'},
	PairU{13601956848995391445, 'd5afbfd481f6f6e1bc01'},
	PairU{3076345003319675583, 'bf95abcbb3efd7d82a'},
	PairU{2359136898918748254, 'dea0c6a4aad0d5de20'},
	PairU{5542048674121246859, '8bd1afa7fffcd3f44c'},
	PairU{18140543185983641493, '95afc0c9a3fd89e0fb01'},
	PairU{12095595652187308684, '8c85899b91858ceea701'},
	PairU{12590579628698639224, 'f8b6e8e6eeaeaeddae01'},
	PairU{14493564399040357302, 'b6c7b1f0fd80df91c901'},
	PairU{13590176164891917038, 'eef5f3bafda680cdbc01'},
	PairU{14863488181145897913, 'b997acc382f8eda2ce01'},
	PairU{10794028799708388741, '85bbd1ef908086e69501'},
	PairU{4823566444556896307, 'b390bbf2ff83b0f842'},
	PairU{9484121148456758217, 'c9d7a3e4f7bd97cf8301'},
	PairU{15355656764698617296, 'd0cbbd85848f908dd501'},
	PairU{8173669839824647064, '98e7c9909dadadb771'},
	PairU{12535116550804629314, 'c2ced1e6ecc2ebfaad01'},
	PairU{17278784277645343305, 'c9f49edd8ed8a4e5ef01'},
]

const signed_data = [
	PairI{71, 'c700'},
	PairI{-57, '47'},
	PairI{58, '3a'},
	PairI{-111, '917f'},
	PairI{30, '1e'},
	PairI{-33, '5f'},
	PairI{-88, 'a87f'},
	PairI{-73, 'b77f'},
	PairI{-38, '5a'},
	PairI{-41, '57'},
	PairI{48, '30'},
	PairI{-50, '4e'},
	PairI{34, '22'},
	PairI{-111, '917f'},
	PairI{-97, '9f7f'},
	PairI{77, 'cd00'},
	PairI{48, '30'},
	PairI{-73, 'b77f'},
	PairI{-63, '41'},
	PairI{-115, '8d7f'},
	PairI{-59, '45'},
	PairI{101, 'e500'},
	PairI{-106, '967f'},
	PairI{-52, '4c'},
	PairI{108, 'ec00'},
	PairI{-43, '55'},
	PairI{-98, '9e7f'},
	PairI{-8, '78'},
	PairI{44, '2c'},
	PairI{-52, '4c'},
	PairI{54, '36'},
	PairI{69, 'c500'},
	PairI{-70, 'ba7f'},
	PairI{-99, '9d7f'},
	PairI{79, 'cf00'},
	PairI{85, 'd500'},
	PairI{-71, 'b97f'},
	PairI{-37, '5b'},
	PairI{-31, '61'},
	PairI{48, '30'},
	PairI{-18, '6e'},
	PairI{86, 'd600'},
	PairI{-116, '8c7f'},
	PairI{-31, '61'},
	PairI{96, 'e000'},
	PairI{54, '36'},
	PairI{69, 'c500'},
	PairI{13, '0d'},
	PairI{-19, '6d'},
	PairI{-69, 'bb7f'},
	PairI{32, '20'},
	PairI{-94, 'a27f'},
	PairI{80, 'd000'},
	PairI{-110, '927f'},
	PairI{100, 'e400'},
	PairI{-4, '7c'},
	PairI{94, 'de00'},
	PairI{45, '2d'},
	PairI{-54, '4a'},
	PairI{-1, '7f'},
	PairI{-121, '877f'},
	PairI{-100, '9c7f'},
	PairI{-33, '5f'},
	PairI{70, 'c600'},
	PairI{-123, '857f'},
	PairI{-23, '69'},
	PairI{-55, '49'},
	PairI{127, 'ff00'},
	PairI{99, 'e300'},
	PairI{79, 'cf00'},
	PairI{62, '3e'},
	PairI{-11, '75'},
	PairI{117, 'f500'},
	PairI{48, '30'},
	PairI{-50, '4e'},
	PairI{63, '3f'},
	PairI{-82, 'ae7f'},
	PairI{38, '26'},
	PairI{84, 'd400'},
	PairI{16, '10'},
	PairI{7, '07'},
	PairI{28, '1c'},
	PairI{15, '0f'},
	PairI{107, 'eb00'},
	PairI{-105, '977f'},
	PairI{116, 'f400'},
	PairI{-51, '4d'},
	PairI{-59, '45'},
	PairI{38, '26'},
	PairI{-81, 'af7f'},
	PairI{-48, '50'},
	PairI{21, '15'},
	PairI{-112, '907f'},
	PairI{-114, '8e7f'},
	PairI{96, 'e000'},
	PairI{108, 'ec00'},
	PairI{50, '32'},
	PairI{-78, 'b27f'},
	PairI{-8, '78'},
	PairI{14, '0e'},
	PairI{36, '24'},
	PairI{12, '0c'},
	PairI{85, 'd500'},
	PairI{78, 'ce00'},
	PairI{-123, '857f'},
	PairI{2, '02'},
	PairI{-8, '78'},
	PairI{-11, '75'},
	PairI{-94, 'a27f'},
	PairI{99, 'e300'},
	PairI{49, '31'},
	PairI{-91, 'a57f'},
	PairI{-66, 'be7f'},
	PairI{-33, '5f'},
	PairI{-19, '6d'},
	PairI{31, '1f'},
	PairI{-56, '48'},
	PairI{21, '15'},
	PairI{20, '14'},
	PairI{-92, 'a47f'},
	PairI{-5, '7b'},
	PairI{119, 'f700'},
	PairI{61, '3d'},
	PairI{55, '37'},
	PairI{83, 'd300'},
	PairI{-26, '66'},
	PairI{101, 'e500'},
	PairI{-81, 'af7f'},
	PairI{20, '14'},
	PairI{-32, '60'},
	PairI{-42, '56'},
	PairI{-121, '877f'},
	PairI{48, '30'},
	PairI{-28, '64'},
	PairI{96, 'e000'},
	PairI{-9, '77'},
	PairI{-97, '9f7f'},
	PairI{-21, '6b'},
	PairI{-30, '62'},
	PairI{62, '3e'},
	PairI{74, 'ca00'},
	PairI{-44, '54'},
	PairI{42, '2a'},
	PairI{62, '3e'},
	PairI{38, '26'},
	PairI{81, 'd100'},
	PairI{-102, '9a7f'},
	PairI{-62, '42'},
	PairI{-56, '48'},
	PairI{-54, '4a'},
	PairI{-104, '987f'},
	PairI{108, 'ec00'},
	PairI{-9, '77'},
	PairI{80, 'd000'},
	PairI{89, 'd900'},
	PairI{-101, '9b7f'},
	PairI{103, 'e700'},
	PairI{-127, '817f'},
	PairI{76, 'cc00'},
	PairI{53, '35'},
	PairI{-65, 'bf7f'},
	PairI{50, '32'},
	PairI{117, 'f500'},
	PairI{13, '0d'},
	PairI{-64, '40'},
	PairI{66, 'c200'},
	PairI{-9, '77'},
	PairI{-103, '997f'},
	PairI{16, '10'},
	PairI{45, '2d'},
	PairI{-5, '7b'},
	PairI{-11, '75'},
	PairI{2, '02'},
	PairI{113, 'f100'},
	PairI{-97, '9f7f'},
	PairI{-66, 'be7f'},
	PairI{-86, 'aa7f'},
	PairI{62, '3e'},
	PairI{-50, '4e'},
	PairI{-57, '47'},
	PairI{-32, '60'},
	PairI{46, '2e'},
	PairI{41, '29'},
	PairI{10, '0a'},
	PairI{20, '14'},
	PairI{-90, 'a67f'},
	PairI{-46, '52'},
	PairI{-127, '817f'},
	PairI{90, 'da00'},
	PairI{107, 'eb00'},
	PairI{-18, '6e'},
	PairI{98, 'e200'},
	PairI{26, '1a'},
	PairI{85, 'd500'},
	PairI{70, 'c600'},
	PairI{-37, '5b'},
	PairI{-55, '49'},
	PairI{92, 'dc00'},
	PairI{-41, '57'},
	PairI{86, 'd600'},
	PairI{-8, '78'},
	PairI{-10, '76'},
	PairI{-71, 'b97f'},
	PairI{-117, '8b7f'},
	PairI{-101, '9b7f'},
	PairI{25, '19'},
	PairI{111, 'ef00'},
	PairI{-48, '50'},
	PairI{-87, 'a97f'},
	PairI{-8, '78'},
	PairI{13, '0d'},
	PairI{-71, 'b97f'},
	PairI{-74, 'b67f'},
	PairI{-69, 'bb7f'},
	PairI{115, 'f300'},
	PairI{-48, '50'},
	PairI{-123, '857f'},
	PairI{5, '05'},
	PairI{-81, 'af7f'},
	PairI{38, '26'},
	PairI{-83, 'ad7f'},
	PairI{-116, '8c7f'},
	PairI{-83, 'ad7f'},
	PairI{84, 'd400'},
	PairI{-29, '63'},
	PairI{-4, '7c'},
	PairI{-49, '4f'},
	PairI{-3, '7d'},
	PairI{-99, '9d7f'},
	PairI{-83, 'ad7f'},
	PairI{66, 'c200'},
	PairI{52, '34'},
	PairI{-17, '6f'},
	PairI{53, '35'},
	PairI{-36, '5c'},
	PairI{-78, 'b27f'},
	PairI{84, 'd400'},
	PairI{48, '30'},
	PairI{-54, '4a'},
	PairI{56, '38'},
	PairI{-73, 'b77f'},
	PairI{-90, 'a67f'},
	PairI{101, 'e500'},
	PairI{-41, '57'},
	PairI{-81, 'af7f'},
	PairI{-70, 'ba7f'},
	PairI{44, '2c'},
	PairI{-77, 'b37f'},
	PairI{-58, '46'},
	PairI{-49, '4f'},
	PairI{-27, '65'},
	PairI{-109, '937f'},
	PairI{5, '05'},
	PairI{76, 'cc00'},
	PairI{61, '3d'},
	PairI{27, '1b'},
	PairI{8787, 'd3c400'},
	PairI{-17881, 'a7f47e'},
	PairI{-31605, '8b897e'},
	PairI{-27049, 'd7ac7e'},
	PairI{3714, '821d'},
	PairI{-2242, 'be6e'},
	PairI{26116, '84cc01'},
	PairI{-27200, 'c0ab7e'},
	PairI{19838, 'fe9a01'},
	PairI{32527, '8ffe01'},
	PairI{31920, 'b0f901'},
	PairI{28648, 'e8df01'},
	PairI{-20914, 'cedc7e'},
	PairI{-11700, 'cca47f'},
	PairI{5055, 'bf27'},
	PairI{4597, 'f523'},
	PairI{-20363, 'f5e07e'},
	PairI{30845, 'fdf001'},
	PairI{-29474, 'de997e'},
	PairI{29595, '9be701'},
	PairI{-12992, 'c09a7f'},
	PairI{32584, 'c8fe01'},
	PairI{24909, 'cdc201'},
	PairI{-7726, 'd243'},
	PairI{-25220, 'fcba7e'},
	PairI{-19006, 'c2eb7e'},
	PairI{-26864, '90ae7e'},
	PairI{26852, 'e4d101'},
	PairI{-28014, '92a57e'},
	PairI{19211, '8b9601'},
	PairI{6328, 'b831'},
	PairI{-6461, 'c34d'},
	PairI{-29068, 'f49c7e'},
	PairI{-25398, 'cab97e'},
	PairI{31065, 'd9f201'},
	PairI{233, 'e901'},
	PairI{-7979, 'd541'},
	PairI{23972, 'a4bb01'},
	PairI{-15692, 'b4857f'},
	PairI{-19202, 'fee97e'},
	PairI{10476, 'ecd100'},
	PairI{30373, 'a5ed01'},
	PairI{-29994, 'd6957e'},
	PairI{29009, 'd1e201'},
	PairI{12224, 'c0df00'},
	PairI{22522, 'faaf01'},
	PairI{-14512, 'd08e7f'},
	PairI{-10081, '9fb17f'},
	PairI{13399, 'd7e800'},
	PairI{13280, 'e0e700'},
	PairI{800, 'a006'},
	PairI{2480, 'b013'},
	PairI{-24927, 'a1bd7e'},
	PairI{-31465, '978a7e'},
	PairI{15847, 'e7fb00'},
	PairI{-28526, '92a17e'},
	PairI{9416, 'c8c900'},
	PairI{30728, '88f001'},
	PairI{25342, 'fec501'},
	PairI{2399, 'df12'},
	PairI{-17174, 'eaf97e'},
	PairI{-29932, '94967e'},
	PairI{-865, '9f79'},
	PairI{-6731, 'b54b'},
	PairI{11834, 'badc00'},
	PairI{26424, 'b8ce01'},
	PairI{29483, 'abe601'},
	PairI{29674, 'eae701'},
	PairI{29302, 'f6e401'},
	PairI{3536, 'd01b'},
	PairI{-1976, 'c870'},
	PairI{-2483, 'cd6c'},
	PairI{6818, 'a235'},
	PairI{7495, 'c73a'},
	PairI{30181, 'e5eb01'},
	PairI{445, 'bd03'},
	PairI{24607, '9fc001'},
	PairI{18354, 'b28f01'},
	PairI{8393, 'c9c100'},
	PairI{-7009, '9f49'},
	PairI{-17596, 'c4f67e'},
	PairI{-28262, '9aa37e'},
	PairI{13564, 'fce900'},
	PairI{6971, 'bb36'},
	PairI{-6278, 'fa4e'},
	PairI{-30326, '8a937e'},
	PairI{-9156, 'bcb87f'},
	PairI{11817, 'a9dc00'},
	PairI{-14692, '9c8d7f'},
	PairI{-1478, 'ba74'},
	PairI{22666, '8ab101'},
	PairI{-12170, 'f6a07f'},
	PairI{2021, 'e50f'},
	PairI{-7431, 'f945'},
	PairI{30262, 'b6ec01'},
	PairI{32557, 'adfe01'},
	PairI{23336, 'a8b601'},
	PairI{31526, 'a6f601'},
	PairI{18786, 'e29201'},
	PairI{28848, 'b0e101'},
	PairI{-5526, 'ea54'},
	PairI{17921, '818c01'},
	PairI{-13761, 'bf947f'},
	PairI{31251, '93f401'},
	PairI{-18544, '90ef7e'},
	PairI{-3191, '8967'},
	PairI{10097, 'f1ce00'},
	PairI{-7666, '8e44'},
	PairI{-18487, 'c9ef7e'},
	PairI{6812, '9c35'},
	PairI{-11996, 'a4a27f'},
	PairI{-19962, '86e47e'},
	PairI{21115, 'fba401'},
	PairI{25267, 'b3c501'},
	PairI{1128, 'e808'},
	PairI{19925, 'd59b01'},
	PairI{-1500, 'a474'},
	PairI{25913, 'b9ca01'},
	PairI{29865, 'a9e901'},
	PairI{-9920, 'c0b27f'},
	PairI{-30417, 'af927e'},
	PairI{-22005, '8bd47e'},
	PairI{-7297, 'ff46'},
	PairI{17461, 'b58801'},
	PairI{23293, 'fdb501'},
	PairI{-32534, 'ea817e'},
	PairI{14725, '85f300'},
	PairI{-26360, '88b27e'},
	PairI{-21339, 'a5d97e'},
	PairI{-24044, '94c47e'},
	PairI{13904, 'd0ec00'},
	PairI{-2029, '9370'},
	PairI{-10100, '8cb17f'},
	PairI{9882, '9acd00'},
	PairI{12336, 'b0e000'},
	PairI{14459, 'fbf000'},
	PairI{27621, 'e5d701'},
	PairI{32327, 'c7fc01'},
	PairI{-30351, 'f1927e'},
	PairI{19835, 'fb9a01'},
	PairI{-3164, 'a467'},
	PairI{-9237, 'ebb77f'},
	PairI{13532, 'dce900'},
	PairI{-28272, '90a37e'},
	PairI{29287, 'e7e401'},
	PairI{2960, '9017'},
	PairI{17445, 'a58801'},
	PairI{-15227, '85897f'},
	PairI{2836, '9416'},
	PairI{28072, 'a8db01'},
	PairI{12460, 'ace100'},
	PairI{-20424, 'b8e07e'},
	PairI{-5743, '9153'},
	PairI{6625, 'e133'},
	PairI{7561, '893b'},
	PairI{-19432, '98e87e'},
	PairI{1819, '9b0e'},
	PairI{20455, 'e79f01'},
	PairI{-25368, 'e8b97e'},
	PairI{-22569, 'd7cf7e'},
	PairI{-20500, 'ecdf7e'},
	PairI{-9091, 'fdb87f'},
	PairI{13550, 'eee900'},
	PairI{2762, 'ca15'},
	PairI{18058, '8a8d01'},
	PairI{31546, 'baf601'},
	PairI{-14136, 'c8917f'},
	PairI{-25615, 'f1b77e'},
	PairI{17927, '878c01'},
	PairI{1689, '990d'},
	PairI{2918, 'e616'},
	PairI{-24013, 'b3c47e'},
	PairI{634, 'fa04'},
	PairI{-22456, 'c8d07e'},
	PairI{12060, '9cde00'},
	PairI{-17556, 'ecf67e'},
	PairI{-24911, 'b1bd7e'},
	PairI{-11677, 'e3a47f'},
	PairI{14630, 'a6f200'},
	PairI{-21202, 'aeda7e'},
	PairI{-31435, 'b58a7e'},
	PairI{-6420, 'ec4d'},
	PairI{-25218, 'feba7e'},
	PairI{9632, 'a0cb00'},
	PairI{-17624, 'a8f67e'},
	PairI{14842, 'faf300'},
	PairI{-13595, 'e5957f'},
	PairI{27098, 'dad301'},
	PairI{31185, 'd1f301'},
	PairI{8754, 'b2c400'},
	PairI{1312, 'a00a'},
	PairI{12, '0c'},
	PairI{-473, 'a77c'},
	PairI{2098, 'b210'},
	PairI{-12244, 'aca07f'},
	PairI{-816, 'd079'},
	PairI{-1225, 'b776'},
	PairI{13591, '97ea00'},
	PairI{2136, 'd810'},
	PairI{-15697, 'af857f'},
	PairI{3544, 'd81b'},
	PairI{-6052, 'dc50'},
	PairI{-9589, '8bb57f'},
	PairI{-11761, '8fa47f'},
	PairI{16861, 'dd8301'},
	PairI{3059, 'f317'},
	PairI{29457, '91e601'},
	PairI{9787, 'bbcc00'},
	PairI{25639, 'a7c801'},
	PairI{3004, 'bc17'},
	PairI{2681, 'f914'},
	PairI{-12251, 'a5a07f'},
	PairI{-719, 'b17a'},
	PairI{4298, 'ca21'},
	PairI{-22438, 'dad07e'},
	PairI{-23226, 'c6ca7e'},
	PairI{-3247, 'd166'},
	PairI{-29117, 'c39c7e'},
	PairI{-17481, 'b7f77e'},
	PairI{23957, '95bb01'},
	PairI{31757, '8df801'},
	PairI{11119, 'efd600'},
	PairI{-1510, '9a74'},
	PairI{10052, 'c4ce00'},
	PairI{23093, 'b5b401'},
	PairI{-10561, 'bfad7f'},
	PairI{5401, '992a'},
	PairI{19166, 'de9501'},
	PairI{-9132, 'd4b87f'},
	PairI{-6059, 'd550'},
	PairI{-26894, 'f2ad7e'},
	PairI{1748, 'd40d'},
	PairI{-5496, '8855'},
	PairI{-18671, '91ee7e'},
	PairI{-19214, 'f2e97e'},
	PairI{-21951, 'c1d47e'},
	PairI{-30256, 'd0937e'},
	PairI{16765, 'fd8201'},
	PairI{-21723, 'a5d67e'},
	PairI{-30595, 'fd907e'},
	PairI{31828, 'd4f801'},
	PairI{-2584, 'e86b'},
	PairI{-30949, '9b8e7e'},
	PairI{-22310, 'dad17e'},
	PairI{21889, '81ab01'},
	PairI{14804, 'd4f300'},
	PairI{4088, 'f81f'},
	PairI{-10743, '89ac7f'},
	PairI{-26876, '84ae7e'},
	PairI{-7980, 'd441'},
	PairI{-15498, 'f6867f'},
	PairI{28570, '9adf01'},
	PairI{27443, 'b3d601'},
	PairI{29549, 'ede601'},
	PairI{21949, 'bdab01'},
	PairI{31972, 'e4f901'},
	PairI{-1610689836, 'd4a5fbff79'},
	PairI{-1828819581, '83dbf99779'},
	PairI{1396236313, '99c0e39905'},
	PairI{491059828, 'f4f493ea01'},
	PairI{-451523301, '9b9ad9a87e'},
	PairI{-614546227, 'cd89fbda7d'},
	PairI{1183392887, 'f7c8a4b404'},
	PairI{512628512, 'a0aeb8f401'},
	PairI{-829518712, '8899baf47c'},
	PairI{1424571991, 'd7fca4a705'},
	PairI{-377531844, 'bca4fdcb7e'},
	PairI{-2048048916, 'ec81b5af78'},
	PairI{-1781899811, 'ddbba9ae79'},
	PairI{-753465411, 'bd8fdc987d'},
	PairI{-123893637, 'fb90f644'},
	PairI{808521482, '8a9ec48103'},
	PairI{594058808, 'b8bca29b02'},
	PairI{-1078535311, 'f1b6dbfd7b'},
	PairI{-631537179, 'e583eed27d'},
	PairI{-1111424379, '858584ee7b'},
	PairI{671205267, '938f87c002'},
	PairI{-1672027781, 'fbc2dbe279'},
	PairI{590553306, 'dac1cc9902'},
	PairI{-1908171085, 'b3bd8ef278'},
	PairI{621854289, 'd1fcc2a802'},
	PairI{392861225, 'a9acaabb01'},
	PairI{1431634128, 'd081d4aa05'},
	PairI{623600501, 'f5c6ada902'},
	PairI{-255554825, 'f79592867f'},
	PairI{1616941558, 'f6a3828306'},
	PairI{-1229050970, 'a6d7f8b57b'},
	PairI{1499620392, 'a8c889cb05'},
	PairI{1715726890, 'aad48fb206'},
	PairI{-417947106, '9ec4dab87e'},
	PairI{-190351805, 'c3ec9da57f'},
	PairI{-236387687, '9985a48f7f'},
	PairI{741239109, 'c5d2b9e102'},
	PairI{1307163713, 'c1f8a6ef04'},
	PairI{-1673941798, 'dad9e6e179'},
	PairI{1705384384, 'c0b398ad06'},
	PairI{-828342709, 'cbfc81f57c'},
	PairI{828595710, 'febb8d8b03'},
	PairI{1118264374, 'b6b89d9504'},
	PairI{-2122511773, 'e394f48b78'},
	PairI{1463843218, '92f381ba05'},
	PairI{2141974662, '86e1affd07'},
	PairI{-1604221749, 'cb8986837a'},
	PairI{-1431106751, 'c196ccd57a'},
	PairI{-665428015, 'd1bfd9c27d'},
	PairI{-385071153, 'cf8fb1c87e'},
	PairI{-895774541, 'b3a1eed47c'},
	PairI{1381938156, 'ece7fa9205'},
	PairI{857769296, 'd08a829903'},
	PairI{155306425, 'b99387ca00'},
	PairI{1422044246, 'd6d88aa605'},
	PairI{558423016, 'e8b7a38a02'},
	PairI{228027580, 'bcd9ddec00'},
	PairI{356371619, 'a399f7a901'},
	PairI{-1772085860, '9cbb80b379'},
	PairI{-1934574707, '8df7c2e578'},
	PairI{1198250140, '9cb1afbb04'},
	PairI{-849861506, 'fec8e0ea7c'},
	PairI{-462860973, 'd39aa5a37e'},
	PairI{879441342, 'beebaca303'},
	PairI{-558187008, '80fceaf57d'},
	PairI{1335418030, 'aeb9e3fc04'},
	PairI{557139109, 'a589d58902'},
	PairI{1090995064, 'f8869d8804'},
	PairI{-1307074103, 'c9c3de907b'},
	PairI{188951315, '93d68cda00'},
	PairI{-1524897325, 'd3d3efa87a'},
	PairI{-708523764, '8c9293ae7d'},
	PairI{223986788, 'e488e7ea00'},
	PairI{-1431958913, 'ff9498d57a'},
	PairI{-396034016, 'a08094c37e'},
	PairI{1150136681, 'e9e2b6a404'},
	PairI{1349151545, 'b9d6a98305'},
	PairI{-2073690705, 'affb97a378'},
	PairI{-1616451762, 'cece9bfd79'},
	PairI{531971183, 'eff8d4fd01'},
	PairI{-938132241, 'eff9d4c07c'},
	PairI{-590957484, 'd4e89ae67d'},
	PairI{-119184036, 'dcca9547'},
	PairI{1222116657, 'b18ae0c604'},
	PairI{645964671, 'ffc682b402'},
	PairI{-974571241, '97f2a4af7c'},
	PairI{-524108653, '93f98a867e'},
	PairI{-1733093001, 'f7b2ccc579'},
	PairI{1449079907, 'e3e8fcb205'},
	PairI{310098077, '9df1ee9301'},
	PairI{-500617726, '82dca4917e'},
	PairI{30820535, 'b791d90e'},
	PairI{1546599379, 'd3f7bce105'},
	PairI{-439073323, 'd58bd1ae7e'},
	PairI{1177334242, 'e2e3b2b104'},
	PairI{-2127666182, 'fac7b98978'},
	PairI{-570346284, 'd4e984f07d'},
	PairI{1556052618, '8af5fde505'},
	PairI{-993557311, 'c1899ea67c'},
	PairI{-554930165, '8be0b1f77d'},
	PairI{-902034296, '8899f0d17c'},
	PairI{-1448565203, 'adcca2cd7a'},
	PairI{290782649, 'b9fbd38a01'},
	PairI{1376218739, 'f3dc9d9005'},
	PairI{1866406765, 'edb6fcf906'},
	PairI{1524349890, 'c2f7eed605'},
	PairI{-2140227152, 'b0f3ba8378'},
	PairI{203976048, 'f0daa1e100'},
	PairI{512089466, 'faba97f401'},
	PairI{-1481127424, '8094dfbd7a'},
	PairI{-1696023128, 'a8fba2d779'},
	PairI{-985922027, '958cf0a97c'},
	PairI{795680718, 'cebfb4fb02'},
	PairI{-756100565, 'aba4bb977d'},
	PairI{-1224829917, 'a3a8fab77b'},
	PairI{219074728, 'a8a1bbe800'},
	PairI{-629824963, 'bdc4d6d37d'},
	PairI{-1876224949, 'cba8ac8179'},
	PairI{146401541, '85d2e7c500'},
	PairI{1710097269, 'f586b8af06'},
	PairI{156723024, 'd0ceddca00'},
	PairI{-1356302298, 'a6f0a1f97a'},
	PairI{-880130888, 'b889a9dc7c'},
	PairI{-1774888330, 'f6b4d5b179'},
	PairI{-1262972701, 'e3a1e2a57b'},
	PairI{-1502181668, 'dc8ddab37a'},
	PairI{1625170515, 'd3c4f88606'},
	PairI{1503705467, 'fbf282cd05'},
	PairI{-838720401, 'efc888f07c'},
	PairI{756645306, 'bafbe5e802'},
	PairI{988137529, 'b99097d703'},
	PairI{54185204, 'f499eb19'},
	PairI{1662854158, '8ec8f49806'},
	PairI{167643376, 'f091f8cf00'},
	PairI{1522174627, 'a395ead505'},
	PairI{1458536356, 'a4ffbdb705'},
	PairI{-716928155, 'e59692aa7d'},
	PairI{-568014405, 'bb9393f17d'},
	PairI{1253387367, 'e7d8d4d504'},
	PairI{746718307, 'e38888e402'},
	PairI{-1729295792, 'd094b4c779'},
	PairI{-1148490108, '84ddaddc7b'},
	PairI{-744194014, 'a280929d7d'},
	PairI{-1801863077, 'db80e7a479'},
	PairI{-1562827802, 'e6c7e4967a'},
	PairI{-818235523, 'fdeeeaf97c'},
	PairI{-2081080403, 'adf7d49f78'},
	PairI{801015649, 'e18efafd02'},
	PairI{451723339, 'cb80b3d701'},
	PairI{877076572, 'dcc09ca203'},
	PairI{164568219, '9bb9bcce00'},
	PairI{194751635, '93d9eedc00'},
	PairI{-1191777396, '8cd7dbc77b'},
	PairI{638074973, 'dd80a1b002'},
	PairI{1140825765, 'a5bdfe9f04'},
	PairI{719627570, 'b2ca92d702'},
	PairI{-108762683, 'c5d3914c'},
	PairI{1700298502, '86fee1aa06'},
	PairI{-354023403, '959098d77e'},
	PairI{-1662424457, 'f7d4a5e779'},
	PairI{-1779805230, 'd2a7a9af79'},
	PairI{312108312, '98cae99401'},
	PairI{-787077562, 'c6ccd8887d'},
	PairI{336286145, 'c1a3ada001'},
	PairI{565707053, 'ad82e08d02'},
	PairI{-613505955, 'ddc8badb7d'},
	PairI{-746574635, 'd5d9809c7d'},
	PairI{-324280602, 'e6bdafe57e'},
	PairI{-723379706, '86b488a77d'},
	PairI{-1540724929, 'bfcea9a17a'},
	PairI{1761092444, 'dcc6e0c706'},
	PairI{-1617846754, '9ebcc6fc79'},
	PairI{1961805726, '9e8fbba707'},
	PairI{-202528688, 'd0d0b69f7f'},
	PairI{1317018748, 'fcb880f404'},
	PairI{1637346847, '9fdcdf8c06'},
	PairI{-664005752, '88a7b0c37d'},
	PairI{549155369, 'a9e4ed8502'},
	PairI{915051381, 'f5a6aab403'},
	PairI{-882683495, '99a38ddb7c'},
	PairI{-98728904, 'b888f650'},
	PairI{-282781627, 'c5b094f97e'},
	PairI{619078172, '9cc499a702'},
	PairI{1389006397, 'bd9caa9605'},
	PairI{-1680472232, 'd88ed8de79'},
	PairI{-1472651910, 'fabae4c17a'},
	PairI{2018201191, 'e79cadc207'},
	PairI{-2115766272, '80f08f8f78'},
	PairI{1598896974, 'cef6b4fa05'},
	PairI{152484912, 'b0f8dac800'},
	PairI{-1552298105, '879fe79b7a'},
	PairI{76120615, 'a784a624'},
	PairI{-370571987, 'ad8aa6cf7e'},
	PairI{432698678, 'b6eaa9ce01'},
	PairI{-1832114620, 'c4ccb09679'},
	PairI{240746611, 'f380e6f200'},
	PairI{-1052806457, 'c7e5fd897c'},
	PairI{-1102673542, 'fa929af27b'},
	PairI{-1998164561, 'afdb99c778'},
	PairI{437455786, 'aa97ccd001'},
	PairI{637003915, '8bd1dfaf02'},
	PairI{-1060201049, 'a7bbba867c'},
	PairI{1317235197, 'fdd38df404'},
	PairI{509843088, '90ad8ef301'},
	PairI{-122140000, 'a095e145'},
	PairI{-261722993, '8fd999837f'},
	PairI{-2062501071, 'b1f6c2a878'},
	PairI{338630976, 'c0b2bca101'},
	PairI{67741227, 'abcca620'},
	PairI{1302523980, 'cce08bed04'},
	PairI{-1867480917, 'ab81c28579'},
	PairI{776700162, '8282aef202'},
	PairI{1907500476, 'bccbc88d07'},
	PairI{-1567520880, '908fc6947a'},
	PairI{387032757, 'b5cdc6b801'},
	PairI{-1117746188, 'f49782eb7b'},
	PairI{-1701024960, 'c0d6f1d479'},
	PairI{-2127550250, 'd6d1c08978'},
	PairI{1311105462, 'b6c397f104'},
	PairI{-341701040, 'd09c88dd7e'},
	PairI{2104989848, '98b1deeb07'},
	PairI{1590192706, 'c2d4a1f605'},
	PairI{-1946109476, 'dcf382e078'},
	PairI{1689481187, 'e3dfcda506'},
	PairI{-393498089, '97e4aec47e'},
	PairI{54452448, 'e0c1fb19'},
	PairI{-511003921, 'efe5aa8c7e'},
	PairI{1536793596, 'fcb7e6dc05'},
	PairI{1519963249, 'f198e3d405'},
	PairI{793517802, 'eabdb0fa02'},
	PairI{901069894, 'c6f8d4ad03'},
	PairI{-1526746287, 'd1e6fea77a'},
	PairI{398347666, '929bf9bd01'},
	PairI{694740715, 'ebcda3cb02'},
	PairI{-756372772, 'dcd5aa977d'},
	PairI{-1812464891, '85f6df9f79'},
	PairI{-867929629, 'e3e391e27c'},
	PairI{-1910851719, 'f9eeeaf078'},
	PairI{-1699416675, '9debd3d579'},
	PairI{-277891402, 'b6edbefb7e'},
	PairI{1131156210, 'f2a5b09b04'},
	PairI{1144313454, 'eeacd3a104'},
	PairI{-1624275774, 'c289bef979'},
	PairI{1663241193, 'e9978c9906'},
	PairI{1906660898, 'a2ac958d07'},
	PairI{-1503022886, 'dae1a6b37a'},
	PairI{-1608446160, 'b09e84817a'},
	PairI{-677366077, 'c3ed80bd7d'},
	PairI{-1370603172, 'dc82b9f27a'},
	PairI{1061928252, 'bcfaaefa03'},
	PairI{-1818041564, 'a4c68b9d79'},
	PairI{990901136, '90e7bfd803'},
	PairI{1737515212, 'ccc1c1bc06'},
	PairI{-927436188, 'e4e4e1c57c'},
	PairI{-1696813956, 'fcd8f2d679'},
	PairI{-1606116365, 'f3b792827a'},
	PairI{-2540322427720969431, 'a9a6ef85e1c6bddf5c'},
	PairI{3037843567655826805, 'f5ead2ea92d4a5942a'},
	PairI{6041003950471360021, '95dcced4e1a2fdead300'},
	PairI{8843802306698213837, 'cdf39bd4f6eadfddfa00'},
	PairI{-7224058460184500729, '87ccb0e0aea3bfdf9b7f'},
	PairI{-5276524690623660565, 'ebfbb1b6809881e3b67f'},
	PairI{7865586331763820127, 'df8ccdb785a38b94ed00'},
	PairI{7617806199745078829, 'adcccbbc8bccf8dbe900'},
	PairI{-9015451631251509835, 'b5dba6ec9dd0abf1827f'},
	PairI{5424495271496989819, 'fbf8f4fad8b6eba3cb00'},
	PairI{8268349867323678312, 'e8bce783d98cc5dff200'},
	PairI{-6296776333015610931, 'cd93928497e9d6cea87f'},
	PairI{7415482293595141532, '9cebbfdfb6bbc5f4e600'},
	PairI{-6166955670678844237, 'b399edfcdccfa4b5aa7f'},
	PairI{789471440916181149, '9de1ffe0fb82b1fa0a'},
	PairI{-7218513110023818091, '95e99ef4a592ace99b7f'},
	PairI{-3773749753308058581, 'abd8eb96ffd2bcd04b'},
	PairI{-2342405984312446314, '96e5bacaafc586bf5f'},
	PairI{7824309355749348443, 'dba8d291f7fce1caec00'},
	PairI{-4893379532346778144, 'e0bbb6badda7ce8bbc7f'},
	PairI{-2112449394162310082, 'bec8ddf7fdcec4d762'},
	PairI{3856595619634880935, 'a7c3a8f586a9d8c235'},
	PairI{9028981872620672023, '97a8fc87cfe5d8a6fd00'},
	PairI{2847422287175818170, 'bab79485beef84c227'},
	PairI{6355660132335479029, 'f5d1dfa7f7e6f599d800'},
	PairI{4455083374148249713, 'f1d0ebc698d7e8e93d'},
	PairI{-2535716587620118949, 'db9cd9beace6d4e75c'},
	PairI{7205322524829226382, '8eabc48493d59cffe300'},
	PairI{8448831201312377775, 'af97d8cbebe691a0f500'},
	PairI{8850640218143925141, '95c784eaabccf2e9fa00'},
	PairI{5982527046032636480, 'c0f4c4efec948d83d300'},
	PairI{8208923558131629487, 'affb95f6bb8ffdf5f100'},
	PairI{-4484711469774604056, 'e8f1db9fded5c6e141'},
	PairI{-5979470863740975152, 'd08fafafe1dda982ad7f'},
	PairI{8565431986913687688, '88f188feaae0a1eff600'},
	PairI{-3493896362524734461, '8398bcedd0f8cbc14f'},
	PairI{273922687615936246, 'f6cd92dabae7cae603'},
	PairI{-2978756232555591187, 'ed93acb0d19fd5d456'},
	PairI{-3446394694547916167, 'f9f4b8dd8fc9fc9550'},
	PairI{6387328730631759352, 'f8cbcfa7b9b496d2d800'},
	PairI{-62821989041134405, 'bbc9fad8bdf7b3907f'},
	PairI{-5722955693957385186, '9ee8fbe2e2c3fec9b07f'},
	PairI{-3992189355246770035, '8dd1aea3ed9db9cc48'},
	PairI{-7429215717708838129, '8f96ddd1a9f587f3987f'},
	PairI{1684447867088912431, 'af80ffc3918d97b017'},
	PairI{4419048529340123694, 'ae949fe496a7e7a93d'},
	PairI{2599353785506738716, '9ce4c8b5efcfb08924'},
	PairI{-3960425873444320581, 'bb95d7f0ebb4ef8449'},
	PairI{5656138666606963108, 'a4d3d6bbe584a9bfce00'},
	PairI{3944792956832413902, 'cef1d5c3b389aedf36'},
	PairI{7464470122701447342, 'aec18ba1f380c8cbe700'},
	PairI{2052987474778287514, '9adbf1d3b2a7ebbe1c'},
	PairI{2423010991394925670, 'e6d8d8a78bf590d021'},
	PairI{2238228741409014760, 'e887848ee8a5f2871f'},
	PairI{-6211278320953137430, 'eacd83f4e9e8c6e6a97f'},
	PairI{-2750458018244304965, 'bbbfb1a794a09aea59'},
	PairI{-5635936467873333755, '85fcf5bdc9b4c8e4b17f'},
	PairI{-8483754184008751867, '85cab2e3cbd0e9a18a7f'},
	PairI{-3709363437631825438, 'e2b388a9b4b3ecc24c'},
	PairI{5110716740488584708, '84bcdbc3c3b9baf6c600'},
	PairI{-3349882385064699251, '8db585f781f7b4c151'},
	PairI{5637155912369669356, 'ec89a6a2f7edcc9dce00'},
	PairI{-3426262981006668988, 'c4969bf08cffddb950'},
	PairI{8785404403908288579, 'c3c0ecf8a1d881f6f900'},
	PairI{-3538458889709326407, 'b9dfb5d4b2ccb7f24e'},
	PairI{8923616810142251375, 'eff2ee81dac6c3ebfb00'},
	PairI{9159289276857505834, 'aab08bf19ba2958eff00'},
	PairI{-5778280777440391568, 'f09cca81f687dbe7af7f'},
	PairI{-5377759596865526605, 'b3c987abba8497afb57f'},
	PairI{-4089601000917788115, 'adb4dad9eab1b49f47'},
	PairI{-2507055343404571362, '9eca80c1abcec99a5d'},
	PairI{-569993810965739236, '9c92dde9e1b4be8b78'},
	PairI{1331218022305811852, '8c9b8bd1cff7dbbc12'},
	PairI{-5779439755263686250, '9693f3959dc5d3e5af7f'},
	PairI{1845000590005876091, 'fbeabecdaec8b0cd19'},
	PairI{2385151848139604305, 'd182e99d80dff08c21'},
	PairI{2787605036686649034, 'cacdb08eaf80e4d726'},
	PairI{-7276726830436499218, 'ee819d8286f0f7819b7f'},
	PairI{-2543542719845913221, 'fbdab684f6abe1d95c'},
	PairI{-6374141538702420700, 'a4aad2ecac81a0c5a77f'},
	PairI{2617408687338066000, 'd0a0ba86d6eab9a924'},
	PairI{2654386888249066281, 'a986a994fbd991eb24'},
	PairI{2921997595792207886, '8ee0f5ddcfaac1c628'},
	PairI{4145633508785338927, 'aff4eca0e3f68fc439'},
	PairI{7032181223721684836, 'e49ee9e089f2d4cbe100'},
	PairI{8300371820353604781, 'ad99dbfe9786b698f300'},
	PairI{7858694681180540595, 'b39d92b0cea5ec87ed00'},
	PairI{-5622710758104924047, 'f1808ee3b1cb87fcb17f'},
	PairI{-7701038245294958103, 'e9b3fcdbc8d09a90957f'},
	PairI{1163598511564577811, '93b8ebb7e6d5fb9210'},
	PairI{-171536181428024305, '8f98ecbe8798a5cf7d'},
	PairI{3579447585051565989, 'a5bf83b9d293b0d631'},
	PairI{7867379394904051495, 'a79eef9386fca297ed00'},
	PairI{536630527899226143, '9fa0c1b8f3d39fb907'},
	PairI{3508229690431827270, 'c6daabc58e8aefd730'},
	PairI{5096346593941490257, 'd1949e819c87f7dcc600'},
	PairI{2922611585456332576, 'a0bec3bc89f8ccc728'},
	PairI{-6554927559093407642, 'e6e0c6fed4838e84a57f'},
	PairI{8629983544958868022, 'b6b4f5f5f789f7e1f700'},
	PairI{-5787331914243643991, 'a9cbe7f19589d1d7af7f'},
	PairI{-3998247270963412067, '9d87b4d0a6e9d7c148'},
	PairI{-5409157020114747574, 'cafeff8d968bb4f7b47f'},
	PairI{1415057224764416882, 'f2c6bb97c7e1d2d113'},
	PairI{-4844991702395721618, 'eed0f3f7f6b6c8e1bc7f'},
	PairI{-1443846670923883028, 'ecbbceaba4a39bfb6b'},
	PairI{1850424091051842564, '84b8e1b782dd81d719'},
	PairI{2481936201139769293, 'cd9ff9c0b3fae6b822'},
	PairI{-7169700919810018527, 'a1e6bac780e086c09c7f'},
	PairI{9040127731630765776, 'd0ed89c0e788bfbafd00'},
	PairI{-7272820501303806738, 'eee9ee849889f0889b7f'},
	PairI{8668207299393572773, 'a59fb5becd93eaa5f800'},
	PairI{3582844041345008027, '9b83fa8fc4b5b4dc31'},
	PairI{5861396459144691389, 'bdfdb4d8e4a0f7abd100'},
	PairI{2771609536681578438, 'c69ff68f8686afbb26'},
	PairI{3591864919434541601, 'a1f4d4e0d1c3b7ec31'},
	PairI{-4531935102467665968, 'd09faac291a1d58d41'},
	PairI{-5080028964182223657, 'd7a9e4e9a79287c0b97f'},
	PairI{-3551864614040605682, '8ed8d1eebbbecfda4e'},
	PairI{-4591896550820938510, 'f2f9e7d7a7cd93a340'},
	PairI{-922574221217393149, '83ac9e8ba4f4969973'},
	PairI{300937269882522802, 'b29998f8a39bc99604'},
	PairI{7872131855186111995, 'fbdb9eabecc6db9fed00'},
	PairI{-1324900659139526206, 'c283bb8aecbbc0ce6d'},
	PairI{6638965462576089153, 'c190bdc79dfd9591dc00'},
	PairI{2174700203147877119, 'ffb5e5ec80ca85971e'},
	PairI{752843520619383432, '8895e4cf97e6a8b90a'},
	PairI{-3510807628900196704, 'a0bdb0a580e2c6a34f'},
	PairI{-6442875855316439363, 'bdddb0ffb9d193cba67f'},
	PairI{-1802299514279593542, 'baf385c4c0c4bcfe66'},
	PairI{8153330444982341041, 'b19bb3ecf6da9c93f100'},
	PairI{1633954117999019776, '808ef9f4cf93bed616'},
	PairI{-4323123947915954171, '85f0f0e3c9b5cb8044'},
	PairI{8446794406616112204, 'ccb8f088a8d8c29cf500'},
	PairI{-8842841274936780126, 'a2b5ecd1e3d6faa3857f'},
	PairI{7162922528771624224, 'a09acd88ed82f4b3e300'},
	PairI{7256629043231079705, '99fac8a4aeb5aedae400'},
	PairI{9097620239540597421, 'ade5e2b981adcfa0fe00'},
	PairI{-7280002181293982082, 'fefc9c9fe5928ffc9a7f'},
	PairI{-6190903940917779093, 'ebca9e98aab5df8aaa7f'},
	PairI{-3001491593507913480, 'f8818fe5c5e9a3ac56'},
	PairI{-8556157385587939139, 'bdd1c8ad8b869ba1897f'},
	PairI{5960960492615612805, '85f3f6d3c4bfe5dcd200'},
	PairI{-3896772997752851503, 'd1e7ed94acb3f8f549'},
	PairI{7357901855678698491, 'fba7aaff8f98a18ee600'},
	PairI{7539467086860256510, 'fe89b8a0eaabe4d0e800'},
	PairI{-1788025794814968433, '8ffbc596b580ea9767'},
	PairI{792640191116149912, '9889b491dec181800b'},
	PairI{5251537772479181262, 'cee3abe9fbb8cdf0c800'},
	PairI{8169316313223349691, 'bb8beddbf6bccfaff100'},
	PairI{-2893347837054065560, 'e8c8fdc58cefb0ec57'},
	PairI{-5678952082115941293, 'd3d88e88b7e59398b17f'},
	PairI{1617000314023815493, 'c5d2d6b8fea6afb816'},
	PairI{4953286460025105580, 'acd9b184abf9e6dec400'},
	PairI{2686683925822690212, 'a4a7a7f3bb99c1a425'},
	PairI{3281306703837989327, 'cf93fc8afee1e2c42d'},
	PairI{2528043057117208223, '9fbdf4e2c2b8da8a23'},
	PairI{-6848037735711187469, 'f3bbdbd3e3beb8fba07f'},
	PairI{5872561561198404427, 'cbc6e08683f4e1bfd100'},
	PairI{2889241843213539345, '91b0c0b7e0c4a98c28'},
	PairI{6394692322857845157, 'a583f0fe97d9a0dfd800'},
	PairI{7767893289178425805, 'cdd3fbd4a5b9c6e6eb00'},
	PairI{-8107645367689656885, 'cb8b91b0e8eff6bd8f7f'},
	PairI{-3194600936394437380, 'fc99d198fceb9fd553'},
	PairI{-4532775902251836953, 'e7b3d09ed28a968c41'},
	PairI{-7271078635291017708, '94fcafcc9490fc8b9b7f'},
	PairI{4901557005331034230, 'f6e0d5a88784f582c400'},
	PairI{-5432329272797841740, 'b4edb6ffeda99fceb47f'},
	PairI{1811240089168032013, '8de2baaebca8b49119'},
	PairI{3648434166598593390, 'eed687b4b5f1f5d032'},
	PairI{-5935045008163866456, 'a8f1bf93aa809fd1ad7f'},
	PairI{-5275542185978577043, 'edfea2b9d3cae0e4b67f'},
	PairI{821061612412789019, '9bc2cb94fbe5bfb20b'},
	PairI{-1531343856302388803, 'bd93fac084dce4df6a'},
	PairI{6203547570860777324, 'ecbe93e3ddb4db8bd600'},
	PairI{6425964547531840549, 'a5b88e97db96e796d900'},
	PairI{4191926494068517598, 'ded5829cfcddad963a'},
	PairI{-5552306137205707020, 'f4dd87fdc4df8ff9b27f'},
	PairI{-1633568702721236735, '81c2ebf0b7bd99aa69'},
	PairI{2031976714474610935, 'f7c1ff90e581c2991c'},
	PairI{5822603727664217699, 'e3fca9badae782e7d000'},
	PairI{-3830059077683584617, '97c3a3affbb1b9ec4a'},
	PairI{-6573916454518439344, 'd084d58a89bab0e2a47f'},
	PairI{-2553681548773257126, 'daa09aff9485e0c75c'},
	PairI{-6206032071712417024, '80b699a7e3d6efefa97f'},
	PairI{3600247828980990210, '8282a2aeb5caa9fb31'},
	PairI{1567284526904680904, 'c8c3fa80939f87e015'},
	PairI{-3172377041564524708, 'dcbec4ffb7fcdcfc53'},
	PairI{-8845873988257394623, 'c1a0b3909a8fc99e857f'},
	PairI{6571515130884375421, 'fd86dfa39bc6ad99db00'},
	PairI{-2015802339231676599, 'c98ec785bece9b8364'},
	PairI{-8485791988481748275, 'cda5c2c5d6a49a9e8a7f'},
	PairI{6078236340301084731, 'bbf090a4f6f78eadd400'},
	PairI{-6147525758390505168, 'b0e2c7dcccbce6d7aa7f'},
	PairI{-827603766902525297, '8fe5ff97a4d8f0c174'},
	PairI{5696576265281547238, 'e6d79cc0e7bd9387cf00'},
	PairI{4556141255674613490, 'f29df397d2caaa9d3f'},
	PairI{-232927486339319927, '89efc7f2f6b59ee27c'},
	PairI{4199347535970806966, 'b6d9cecfdb8ac5a33a'},
	PairI{6485808696913935799, 'b7fbc6a5d8948e81da00'},
	PairI{-1991599610167483452, 'c4c7a4e9cdd69aae64'},
	PairI{762388058533688319, 'ffe7998fe6fca2ca0a'},
	PairI{6473112946808070161, '91c0e8cbddbdc7ead900'},
	PairI{2681377212024914560, '8085cabce2cb8a9b25'},
	PairI{7238057873582600864, 'a0e5f09498e9afb9e400'},
	PairI{851217854692026439, 'c7d08dfb8fc488e80b'},
	PairI{-6493158617924639365, 'fbcad7a2bad4eaf1a57f'},
	PairI{-2672965747832870286, 'f2d48abd87faedf35a'},
	PairI{-7478508143247252325, '9bc986fff68ec09b987f'},
	PairI{3776760975322363109, 'e5e996fd8c83f0b434'},
	PairI{8920698040007351562, '8abaa5f9a7f3abe6fb00'},
	PairI{77897111127919137, 'a194bcaea8e0af8a01'},
	PairI{3644048037985052406, 'f6cdbb84a3cc90c932'},
	PairI{-4291837239673002165, 'cbcefdbbd09895b844'},
	PairI{1431719763524378113, '81bc9b96a1b19fef13'},
	PairI{-3480526772758105880, 'e8e190f4f3eaabd94f'},
	PairI{-6779538613528389993, '97ade2fcceb18ff5a17f'},
	PairI{3328552264639739130, 'fae9b1b1e394d9982e'},
	PairI{-3182545557078683548, 'e4a0cfe6d7f5d4ea53'},
	PairI{2954022553703925616, 'f08e8fe0c8fbb2ff28'},
	PairI{-2504855703159012186, 'a689e2fca5e0bd9e5d'},
	PairI{6933097952413144173, 'eda8cbb4acfbd39be000'},
	PairI{-8522800211221181049, '879ba6b4d1cbbbdc897f'},
	PairI{1315442208010158808, 'd88581d8fff6d8a012'},
	PairI{4103161484643623377, 'd1c3fde197f4d6f838'},
	PairI{-1563953070423669007, 'f18d90c0ff9eeea56a'},
	PairI{-4594607936156968693, '8bb2b199ce8dab9e40'},
	PairI{262231303773617553, '91fba7f3b3bfe8d103'},
	PairI{5995002221617381565, 'bd81bab99fd8a199d300'},
	PairI{-3920918428027768861, 'e3dffc80ebae86cb49'},
	PairI{-466253085031555368, 'd8bd8edaa1a9e2c379'},
	PairI{5045167886354394053, 'c5d7f292e5ae8282c600'},
	PairI{7600297399029821247, 'bfdefd83f6c6ebbce900'},
	PairI{4132753277204215288, 'f8ebdac0d9a69fad39'},
	PairI{-7496317180987688968, 'f8efc8bcdbe7eefb977f'},
	PairI{-5956359947341419360, 'a0d9e682fac5b0abad7f'},
	PairI{-7322751059733054785, 'bff5fb8fd79697b09a7f'},
	PairI{-4234243536084555642, '86d1bed981bebc9e45'},
	PairI{-2791206811152809932, 'b480d8b49986e9a159'},
	PairI{-973503881282136363, 'd5addbc58becdabe72'},
	PairI{-7897285947534482093, 'd3ca97bcdb88cdb3927f'},
	PairI{-5293955917046730683, 'c5e0e8dec5e485c4b67f'},
	PairI{8714610830351436396, 'ec94fa968d8ca1f8f800'},
	PairI{-573796989413515899, '85bbc686dad5dd8478'},
	PairI{3623670621805681668, '84b8fc9db4a7f7a432'},
	PairI{6890775738228699463, 'c7e281c0e680bdd0df00'},
	PairI{6914526363952000407, '97dbba8bfea2d5fadf00'},
	PairI{-8276697308179582507, 'd5e398c4e6f4d0918d7f'},
	PairI{-4589574099668352347, 'a585bdb7c4d5a3a740'},
	PairI{3741708248961789441, '81dc90c4bffacdf633'},
	PairI{1990638717472956861, 'bdc3f5f8ddeb8ad01b'},
	PairI{-2085412304362508580, 'dcddf9a1ee91c88763'},
	PairI{1532600219281232624, 'f0a5c6a7f8f8b8a215'},
	PairI{-7621567507177824854, 'aabb86b8ba97b09d967f'},
	PairI{-3395109590119605005, 'f3b998db8dba89f150'},
	PairI{-7772703118602814481, 'eff7eeb89ff6f390947f'},
	PairI{-4518171459184852569, 'a783eaa7f0df8ea641'},
]
