// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { isMultiRegionAwsKmsArn, 
// getRegionFromIdentifier,
parseAwsKmsKeyArn, } from '@aws-crypto/kms-keyring';
import { constructArnInOtherRegion, mrkAwareAwsKmsKeyIdCompare, } from '@aws-crypto/kms-keyring';
import { needs, readOnlyProperty } from '@aws-crypto/material-management';
// an abstract class defining common behavior for operations that SRK and MRK compatibility
// configs should perform
export class KmsKeyConfig {
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
    //# `KMS Key ARN` and `KMS MRKey ARN` MUST take an additional argument
    //# that is a KMS ARN.
    constructor(config) {
        readOnlyProperty(this, '_config', config);
        /* Precondition: config must be a string or object */
        const configType = typeof config;
        needs(!!config && (configType === 'object' || configType === 'string'), 'Config must be a `discovery` or an object.');
        if (configType === 'string') {
            /* Precondition: Only `discovery` is a valid string value */
            needs(config === 'discovery', 'Unexpected config shape');
        }
        else if ('identifier' in config ||
            'mrkIdentifier' in config) {
            const arn = 'identifier' in config
                ? config.identifier
                : config.mrkIdentifier;
            /* Precondition: ARN must be a string */
            needs(typeof arn === 'string', 'ARN must be a string');
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
            //# To be clear, an KMS ARN for a Multi-Region Key MAY be provided to the `KMS Key ARN` configuration,
            //# and a KMS ARN for non Multi-Region Key MAY be provided to the `KMS MRKey ARN` configuration.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
            //# This ARN MUST NOT be an Alias.
            //# This ARN MUST be a valid
            //# [AWS KMS Key ARN](./aws-kms/aws-kms-key-arn.md#a-valid-aws-kms-arn).
            const parsedArn = parseAwsKmsKeyArn(arn);
            needs(parsedArn && parsedArn.ResourceType === 'key', `${arn} must be a well-formed AWS KMS non-alias resource arn`);
            readOnlyProperty(this, '_parsedArn', parsedArn);
            readOnlyProperty(this, '_arn', arn);
        }
        else if ('region' in config) {
            readOnlyProperty(this, '_mrkRegion', config.region);
        }
        else {
            needs(false, 'Unexpected config shape');
        }
        Object.freeze(this);
    }
    getRegion() {
        if (this._config === 'discovery') {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is Discovery,
            //# a new DynamoDb client MUST be created with the default configuration.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is Discovery,
            //# a new AWS KMS client MUST be created with the default configuration.
            return undefined;
        }
        else if ('identifier' in this._config ||
            'mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
            //# a new DynamoDb client MUST be created with the region of the supplied KMS ARN.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
            //# a new AWS KMS client MUST be created with the region of the supplied KMS ARN.
            return this._parsedArn.Region;
        }
        else if ('region' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
            //# a new DynamoDb client MUST be created with the region configured in the MRDiscovery.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
            //# a new AWS KMS client MUST be created with the region configured in the MRDiscovery.
            return this._mrkRegion;
        }
        else {
            needs(false, 'Unexpected configuration state');
        }
    }
    isCompatibleWithArn(otherArn) {
        if (this._config === 'discovery' || 'region' in this._config) {
            // This may result in the function being called twice.
            // However this is the most correct behavior
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `Discovery` or `MRDiscovery`,
            //# the `kms-arn` field of DDB response item MUST NOT be an Alias
            //# or the operation MUST fail.
            const parsedArn = parseAwsKmsKeyArn(otherArn);
            needs(parsedArn && parsedArn.ResourceType === 'key', `${otherArn} must be a well-formed AWS KMS non-alias resource arn`);
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# If the [AWS KMS Configuration](#aws-kms-configuration) is Discovery or MRDiscovery,
            //# no comparison is ever made between ARNs.
            return true;
        }
        else if ('identifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# For two ARNs to be compatible:
            //#
            //# If the [AWS KMS Configuration](#aws-kms-configuration) designates single region ARN compatibility,
            //# then two ARNs are compatible if they are exactly equal.
            return this._arn === otherArn;
        }
        else if ('mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# If the [AWS KMS Configuration](#aws-kms-configuration) designates MRK ARN compatibility,
            //# then two ARNs are compatible if they are equal in all parts other than the region.
            //# That is, they are compatible if [AWS KMS MRK Match for Decrypt](aws-kms/aws-kms-mrk-match-for-decrypt.md#implementation) returns true.
            return mrkAwareAwsKmsKeyIdCompare(this._arn, otherArn);
        }
        else {
            needs(false, 'Unexpected configuration state');
        }
    }
    getCompatibleArnArn(otherArn) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
        //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `KMS Key ARN` or `KMS MRKey ARN`,
        //# the `kms-arn` field of the DDB response item MUST be
        //# [compatible with](#aws-key-arn-compatibility) the configured KMS Key in
        //# the [AWS KMS Configuration](#aws-kms-configuration) for this keystore,
        //# or the operation MUST fail.
        //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `Discovery` or `MRDiscovery`,
        //# the `kms-arn` field of DDB response item MUST NOT be an Alias
        //# or the operation MUST fail.
        needs(this.isCompatibleWithArn(otherArn), 'KMS ARN from DDB response item MUST be compatible with the configured KMS Key in the AWS KMS Configuration for this keystore');
        if (this._config == 'discovery') {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# - `KeyId`, if the KMS Configuration is Discovery, MUST be the `kms-arn` attribute value of the AWS DDB response item.
            return otherArn;
        }
        else if ('region' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# If the KMS Configuration is MRDiscovery, `KeyId` MUST be the `kms-arn` attribute value of the AWS DDB response item, with the region replaced by the configured region.
            const parsedArn = parseAwsKmsKeyArn(otherArn);
            needs(parsedArn, 'KMS ARN from the keystore is not an ARN:' + otherArn);
            return isMultiRegionAwsKmsArn(parsedArn)
                ? constructArnInOtherRegion(parsedArn, this._mrkRegion)
                : otherArn;
        }
        else if ('identifier' in this._config ||
            'mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# Otherwise, it MUST BE the Keystore's configured KMS Key.
            // In this case, the equality condition has already been satisfied.
            // In the SRK case this is strict equality,
            // in the MKR case this is functional (everything but region)
            return this._arn;
        }
        else {
            // This is for completeness.
            // It should should be impossible to get here.
            needs(false, 'Unexpected configuration state');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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