/*
 * Copyright © 2014 Red Hat, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <config.h>

#include <valgrind/valgrind.h>

#include <fcntl.h>
#include <unistd.h>

#include "litest.h"
#include "litest-runner.h"
#include "util-list.h"
#include "util-strings.h"
#include "util-time.h"
#include "util-prop-parsers.h"
#include "util-macros.h"
#include "util-bits.h"
#include "util-range.h"
#include "util-ratelimit.h"
#include "util-stringbuf.h"
#include "util-matrix.h"
#include "util-input-event.h"

#include "litest.h"

#define  TEST_VERSIONSORT
#include "libinput-versionsort.h"

START_TEST(array_for_each)
{
	int ai[6];
	char ac[10];
	struct as {
		int a;
		char b;
		int *ptr;
	} as[32];

	for (size_t i = 0; i < 6; i++)
		ai[i] = 20 + i;
	for (size_t i = 0; i < 10; i++)
		ac[i] = 100 + i;
	for (size_t i = 0; i < 32; i++) {
		as[i].a = 10 + i;
		as[i].b = 20 + i;
		as[i].ptr = (int*)0xab + i;
	}

	int iexpected = 20;
	ARRAY_FOR_EACH(ai, entry) {
		litest_assert_int_eq(*entry, iexpected);
		++iexpected;
	}
	litest_assert_int_eq(iexpected, 26);

	int cexpected = 100;
	ARRAY_FOR_EACH(ac, entry) {
		litest_assert_int_eq(*entry, cexpected);
		++cexpected;
	}
	litest_assert_int_eq(cexpected, 110);

	struct as sexpected = {
		.a = 10,
		.b = 20,
		.ptr = (int*)0xab,
	};
	ARRAY_FOR_EACH(as, entry) {
		litest_assert_int_eq(entry->a, sexpected.a);
		litest_assert_int_eq(entry->b, sexpected.b);
		litest_assert_ptr_eq(entry->ptr, sexpected.ptr);
		++sexpected.a;
		++sexpected.b;
		++sexpected.ptr;
	}
	litest_assert_int_eq(sexpected.a, 42);
}
END_TEST

START_TEST(bitfield_helpers)
{
	/* This value has a bit set on all of the word boundaries we want to
	 * test: 0, 1, 7, 8, 31, 32, and 33
	 */
	unsigned char read_bitfield[] = { 0x83, 0x1, 0x0, 0x80, 0x3 };
	unsigned char write_bitfield[ARRAY_LENGTH(read_bitfield)] = {0};
	size_t i;

	/* Now check that the bitfield we wrote to came out to be the same as
	 * the bitfield we were writing from */
	for (i = 0; i < ARRAY_LENGTH(read_bitfield) * 8; i++) {
		switch (i) {
		case 0:
		case 1:
		case 7:
		case 8:
		case 31:
		case 32:
		case 33:
			litest_assert(bit_is_set(read_bitfield, i));
			set_bit(write_bitfield, i);
			break;
		default:
			litest_assert(!bit_is_set(read_bitfield, i));
			clear_bit(write_bitfield, i);
			break;
		}
	}

	litest_assert_int_eq(memcmp(read_bitfield,
				write_bitfield,
				sizeof(read_bitfield)),
			 0);
}
END_TEST

START_TEST(matrix_helpers)
{
	struct matrix m1, m2, m3;
	float f[6] = { 1, 2, 3, 4, 5, 6 };
	int x, y;
	int row, col;

	matrix_init_identity(&m1);

	for (row = 0; row < 3; row++) {
		for (col = 0; col < 3; col++) {
			litest_assert_int_eq(m1.val[row][col],
					 (row == col) ? 1 : 0);
		}
	}
	litest_assert(matrix_is_identity(&m1));

	matrix_from_farray6(&m2, f);
	litest_assert_int_eq(m2.val[0][0], 1);
	litest_assert_int_eq(m2.val[0][1], 2);
	litest_assert_int_eq(m2.val[0][2], 3);
	litest_assert_int_eq(m2.val[1][0], 4);
	litest_assert_int_eq(m2.val[1][1], 5);
	litest_assert_int_eq(m2.val[1][2], 6);
	litest_assert_int_eq(m2.val[2][0], 0);
	litest_assert_int_eq(m2.val[2][1], 0);
	litest_assert_int_eq(m2.val[2][2], 1);

	x = 100;
	y = 5;
	matrix_mult_vec(&m1, &x, &y);
	litest_assert_int_eq(x, 100);
	litest_assert_int_eq(y, 5);

	matrix_mult(&m3, &m1, &m1);
	litest_assert(matrix_is_identity(&m3));

	matrix_init_scale(&m2, 2, 4);
	litest_assert_int_eq(m2.val[0][0], 2);
	litest_assert_int_eq(m2.val[0][1], 0);
	litest_assert_int_eq(m2.val[0][2], 0);
	litest_assert_int_eq(m2.val[1][0], 0);
	litest_assert_int_eq(m2.val[1][1], 4);
	litest_assert_int_eq(m2.val[1][2], 0);
	litest_assert_int_eq(m2.val[2][0], 0);
	litest_assert_int_eq(m2.val[2][1], 0);
	litest_assert_int_eq(m2.val[2][2], 1);

	matrix_mult_vec(&m2, &x, &y);
	litest_assert_int_eq(x, 200);
	litest_assert_int_eq(y, 20);

	matrix_init_translate(&m2, 10, 100);
	litest_assert_int_eq(m2.val[0][0], 1);
	litest_assert_int_eq(m2.val[0][1], 0);
	litest_assert_int_eq(m2.val[0][2], 10);
	litest_assert_int_eq(m2.val[1][0], 0);
	litest_assert_int_eq(m2.val[1][1], 1);
	litest_assert_int_eq(m2.val[1][2], 100);
	litest_assert_int_eq(m2.val[2][0], 0);
	litest_assert_int_eq(m2.val[2][1], 0);
	litest_assert_int_eq(m2.val[2][2], 1);

	matrix_mult_vec(&m2, &x, &y);
	litest_assert_int_eq(x, 210);
	litest_assert_int_eq(y, 120);

	matrix_to_farray6(&m2, f);
	litest_assert_int_eq(f[0], 1);
	litest_assert_int_eq(f[1], 0);
	litest_assert_int_eq(f[2], 10);
	litest_assert_int_eq(f[3], 0);
	litest_assert_int_eq(f[4], 1);
	litest_assert_int_eq(f[5], 100);
}
END_TEST

START_TEST(ratelimit_helpers)
{
	struct ratelimit rl;
	unsigned int i, j;

	/* 10 attempts every 1000ms */
	ratelimit_init(&rl, ms2us(1000), 10);

	for (j = 0; j < 3; ++j) {
		/* a burst of 9 attempts must succeed */
		for (i = 0; i < 9; ++i) {
			litest_assert_enum_eq(ratelimit_test(&rl),
					 RATELIMIT_PASS);
		}

		/* the 10th attempt reaches the threshold */
		litest_assert_enum_eq(ratelimit_test(&rl), RATELIMIT_THRESHOLD);

		/* ..then further attempts must fail.. */
		litest_assert_enum_eq(ratelimit_test(&rl), RATELIMIT_EXCEEDED);

		/* ..regardless of how often we try. */
		for (i = 0; i < 100; ++i) {
			litest_assert_enum_eq(ratelimit_test(&rl),
					 RATELIMIT_EXCEEDED);
		}

		/* ..even after waiting 20ms */
		msleep(100);
		for (i = 0; i < 100; ++i) {
			litest_assert_enum_eq(ratelimit_test(&rl),
					 RATELIMIT_EXCEEDED);
		}

		/* but after 1000ms the counter is reset */
		msleep(950); /* +50ms to account for time drifts */
	}
}
END_TEST

struct parser_test {
	char *tag;
	int expected_value;
};

START_TEST(dpi_parser)
{
	struct parser_test tests[] = {
		{ "450 *1800 3200", 1800 },
		{ "*450 1800 3200", 450 },
		{ "450 1800 *3200", 3200 },
		{ "450 1800 3200", 3200 },
		{ "450 1800 failboat", 0 },
		{ "450 1800 *failboat", 0 },
		{ "0 450 1800 *3200", 0 },
		{ "450@37 1800@12 *3200@6", 3200 },
		{ "450@125 1800@125   *3200@125  ", 3200 },
		{ "450@125 *1800@125  3200@125", 1800 },
		{ "*this @string fails", 0 },
		{ "12@34 *45@", 0 },
		{ "12@a *45@", 0 },
		{ "12@a *45@25", 0 },
		{ "                                      * 12, 450, 800", 0 },
		{ "                                      *12, 450, 800", 12 },
		{ "*12, *450, 800", 12 },
		{ "*-23412, 450, 800", 0 },
		{ "112@125, 450@125, 800@125, 900@-125", 0 },
		{ "", 0 },
		{ "   ", 0 },
		{ "* ", 0 },
		{ NULL, 0 }
	};
	int i, dpi;

	for (i = 0; tests[i].tag != NULL; i++) {
		dpi = parse_mouse_dpi_property(tests[i].tag);
		litest_assert_int_eq(dpi, tests[i].expected_value);
	}

	dpi = parse_mouse_dpi_property(NULL);
	litest_assert_int_eq(dpi, 0);
}
END_TEST

START_TEST(wheel_click_parser)
{
	struct parser_test tests[] = {
		{ "1", 1 },
		{ "10", 10 },
		{ "-12", -12 },
		{ "360", 360 },

		{ "0", 0 },
		{ "-0", 0 },
		{ "a", 0 },
		{ "10a", 0 },
		{ "10-", 0 },
		{ "sadfasfd", 0 },
		{ "361", 0 },
		{ NULL, 0 }
	};

	int i, angle;

	for (i = 0; tests[i].tag != NULL; i++) {
		angle = parse_mouse_wheel_click_angle_property(tests[i].tag);
		litest_assert_int_eq(angle, tests[i].expected_value);
	}
}
END_TEST

START_TEST(wheel_click_count_parser)
{
	struct parser_test tests[] = {
		{ "1", 1 },
		{ "10", 10 },
		{ "-12", -12 },
		{ "360", 360 },

		{ "0", 0 },
		{ "-0", 0 },
		{ "a", 0 },
		{ "10a", 0 },
		{ "10-", 0 },
		{ "sadfasfd", 0 },
		{ "361", 0 },
		{ NULL, 0 }
	};

	int i, angle;

	for (i = 0; tests[i].tag != NULL; i++) {
		angle = parse_mouse_wheel_click_count_property(tests[i].tag);
		litest_assert_int_eq(angle, tests[i].expected_value);
	}

	angle = parse_mouse_wheel_click_count_property(NULL);
	litest_assert_int_eq(angle, 0);
}
END_TEST

START_TEST(dimension_prop_parser)
{
	struct parser_test_dimension {
		char *tag;
		bool success;
		size_t x, y;
	} tests[] = {
		{ "10x10", true, 10, 10 },
		{ "1x20", true, 1, 20 },
		{ "1x8000", true, 1, 8000 },
		{ "238492x428210", true, 238492, 428210 },
		{ "0x0", false, 0, 0 },
		{ "-10x10", false, 0, 0 },
		{ "-1", false, 0, 0 },
		{ "1x-99", false, 0, 0 },
		{ "0", false, 0, 0 },
		{ "100", false, 0, 0 },
		{ "", false, 0, 0 },
		{ "abd", false, 0, 0 },
		{ "xabd", false, 0, 0 },
		{ "0xaf", false, 0, 0 },
		{ "0x0x", false, 0, 0 },
		{ "x10", false, 0, 0 },
		{ NULL, false, 0, 0 }
	};
	int i;
	size_t x, y;
	bool success;

	for (i = 0; tests[i].tag != NULL; i++) {
		x = y = 0xad;
		success = parse_dimension_property(tests[i].tag, &x, &y);
		litest_assert(success == tests[i].success);
		if (success) {
			litest_assert_int_eq(x, tests[i].x);
			litest_assert_int_eq(y, tests[i].y);
		} else {
			litest_assert_int_eq(x, 0xadU);
			litest_assert_int_eq(y, 0xadU);
		}
	}

	success = parse_dimension_property(NULL, &x, &y);
	litest_assert(success == false);
}
END_TEST

START_TEST(reliability_prop_parser)
{
	struct parser_test_reliability {
		char *tag;
		bool success;
		enum switch_reliability reliability;
	} tests[] = {
		{ "reliable", true, RELIABILITY_RELIABLE },
		{ "unreliable", true, RELIABILITY_UNRELIABLE },
		{ "write_open", true, RELIABILITY_WRITE_OPEN },
		{ "", false, 0 },
		{ "0", false, 0 },
		{ "1", false, 0 },
		{ NULL, false, 0, }
	};
	enum switch_reliability r;
	bool success;
	int i;

	for (i = 0; tests[i].tag != NULL; i++) {
		r = 0xaf;
		success = parse_switch_reliability_property(tests[i].tag, &r);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(r, tests[i].reliability);
		else
			litest_assert_int_eq(r, 0xafU);
	}

	success = parse_switch_reliability_property(NULL, &r);
	litest_assert(success == true);
	litest_assert_enum_eq(r, RELIABILITY_RELIABLE);

	success = parse_switch_reliability_property("foo", NULL);
	litest_assert(success == false);
}
END_TEST

START_TEST(calibration_prop_parser)
{
#define DEFAULT_VALUES { 1.0, 2.0, 3.0, 4.0, 5.0, 6.0 }
	const float untouched[6] = DEFAULT_VALUES;
	struct parser_test_calibration {
		char *prop;
		bool success;
		float values[6];
	} tests[] = {
		{ "", false, DEFAULT_VALUES },
		{ "banana", false, DEFAULT_VALUES },
		{ "1 2 3 a 5 6", false, DEFAULT_VALUES },
		{ "2", false, DEFAULT_VALUES },
		{ "2 3 4 5 6", false, DEFAULT_VALUES },
		{ "1 2 3 4 5 6", true, DEFAULT_VALUES },
		{ "6.00012 3.244 4.238 5.2421 6.0134 8.860", true,
			{ 6.00012, 3.244, 4.238, 5.2421, 6.0134, 8.860 }},
		{ "0xff 2 3 4 5 6", false, DEFAULT_VALUES },
		{ NULL, false, DEFAULT_VALUES }
	};
	bool success;
	float calibration[6];
	int rc;
	int i;

	for (i = 0; tests[i].prop != NULL; i++) {
		memcpy(calibration, untouched, sizeof(calibration));

		success = parse_calibration_property(tests[i].prop,
						     calibration);
		litest_assert_int_eq(success, tests[i].success);
		if (success)
			rc = memcmp(tests[i].values,
				    calibration,
				    sizeof(calibration));
		else
			rc = memcmp(untouched,
				    calibration,
				    sizeof(calibration));
		litest_assert_int_eq(rc, 0);
	}

	memcpy(calibration, untouched, sizeof(calibration));

	success = parse_calibration_property(NULL, calibration);
	litest_assert(success == false);
	rc = memcmp(untouched, calibration, sizeof(calibration));
	litest_assert_int_eq(rc, 0);
}
END_TEST

START_TEST(range_prop_parser)
{
	struct parser_test_range {
		char *tag;
		bool success;
		int hi, lo;
	} tests[] = {
		{ "10:8", true, 10, 8 },
		{ "100:-1", true, 100, -1 },
		{ "-203813:-502023", true, -203813, -502023 },
		{ "238492:28210", true, 238492, 28210 },
		{ "none", true, 0, 0 },
		{ "0:0", false, 0, 0 },
		{ "", false, 0, 0 },
		{ "abcd", false, 0, 0 },
		{ "10:30:10", false, 0, 0 },
		{ NULL, false, 0, 0 }
	};
	int i;
	int hi, lo;
	bool success;

	for (i = 0; tests[i].tag != NULL; i++) {
		hi = lo = 0xad;
		success = parse_range_property(tests[i].tag, &hi, &lo);
		litest_assert(success == tests[i].success);
		if (success) {
			litest_assert_int_eq(hi, tests[i].hi);
			litest_assert_int_eq(lo, tests[i].lo);
		} else {
			litest_assert_int_eq(hi, 0xad);
			litest_assert_int_eq(lo, 0xad);
		}
	}

	success = parse_range_property(NULL, NULL, NULL);
	litest_assert(success == false);
}
END_TEST

START_TEST(boolean_prop_parser)
{
	struct parser_test_range {
		char *tag;
		bool success;
		bool b;
	} tests[] = {
		{ "0", true, false },
		{ "1", true, true },
		{ "-1", false, false },
		{ "2", false, false },
		{ "abcd", false, false },
		{ NULL, false, false }
	};
	int i;
	bool success, b;

	for (i = 0; tests[i].tag != NULL; i++) {
		b = false;
		success = parse_boolean_property(tests[i].tag, &b);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(b, tests[i].b);
		else
			litest_assert_int_eq(b, false);
	}

	success = parse_boolean_property(NULL, NULL);
	litest_assert(success == false);
}
END_TEST

START_TEST(evcode_prop_parser)
{
	struct parser_test_tuple {
		const char *prop;
		bool success;
		size_t nevents;
		struct input_event events[20];
	} tests[] = {
		{ "+EV_KEY", true, 1, {{ .type = EV_KEY, .code = 0xffff, .value = 1 }} },
		{ "-EV_ABS;", true, 1, {{ .type = EV_ABS, .code = 0xffff, .value = 0 }} },
		{ "+ABS_X;", true, 1, {{ .type = EV_ABS, .code = ABS_X, .value = 1 }} },
		{ "-SW_TABLET_MODE;", true, 1, {{ .type = EV_SW, .code = SW_TABLET_MODE, .value = 0 }} },
		{ "+EV_SW", true, 1, {{ .type = EV_SW, .code = 0xffff, .value = 1 }} },
		{ "-ABS_Y", true, 1, {{ .type = EV_ABS, .code = ABS_Y, .value = 0 }} },
		{ "+EV_ABS:0x00", true, 1, {{ .type = EV_ABS, .code = ABS_X, .value = 1 }} },
		{ "-EV_ABS:01", true, 1, {{ .type = EV_ABS, .code = ABS_Y, .value = 0 }} },
		{ "+ABS_TILT_X;-ABS_TILT_Y;", true, 2,
			{{ .type = EV_ABS, .code = ABS_TILT_X, .value = 1 },
			 { .type = EV_ABS, .code = ABS_TILT_Y, .value = 0}} },
		{ "+BTN_TOOL_DOUBLETAP;+EV_KEY;-KEY_A", true, 3,
			{{ .type = EV_KEY, .code = BTN_TOOL_DOUBLETAP, .value = 1 } ,
			 { .type = EV_KEY, .code = 0xffff, .value = 1 },
			 { .type = EV_KEY, .code = KEY_A, .value = 0 }} },
		{ "+REL_Y;-ABS_Z;+BTN_STYLUS", true, 3,
			{{ .type = EV_REL, .code = REL_Y, .value = 1},
			 { .type = EV_ABS, .code = ABS_Z, .value = 0},
			 { .type = EV_KEY, .code = BTN_STYLUS, .value = 1 }} },
		{ "-REL_Y;+EV_KEY:0x123;-BTN_STYLUS", true, 3,
			{{ .type = EV_REL, .code = REL_Y, .value = 0 },
			 { .type = EV_KEY, .code = 0x123, .value = 1 },
			 { .type = EV_KEY, .code = BTN_STYLUS, .value = 0 }} },
		{ .prop = "", .success = false },
		{ .prop = "+", .success = false },
		{ .prop = "-", .success = false },
		{ .prop = "!", .success = false },
		{ .prop = "+EV_FOO", .success = false },
		{ .prop = "+EV_KEY;-EV_FOO", .success = false },
		{ .prop = "+BTN_STYLUS;-EV_FOO", .success = false },
		{ .prop = "-BTN_UNKNOWN", .success = false },
		{ .prop = "+BTN_UNKNOWN;+EV_KEY", .success = false },
		{ .prop = "-PR_UNKNOWN", .success = false },
		{ .prop = "-BTN_STYLUS;+PR_UNKNOWN;-ABS_X", .success = false },
		{ .prop = "-EV_REL:0xffff", .success = false },
		{ .prop = "-EV_REL:0x123.", .success = false },
		{ .prop = "-EV_REL:ffff", .success = false },
		{ .prop = "-EV_REL:blah", .success = false },
		{ .prop = "+KEY_A:0x11", .success = false },
		{ .prop = "+EV_KEY:0x11 ", .success = false },
		{ .prop = "+EV_KEY:0x11not", .success = false },
		{ .prop = "none", .success = false },
		{ .prop = NULL },
	};
	struct parser_test_tuple *t;

	for (int i = 0; tests[i].prop; i++) {
		bool success;
		struct input_event events[32];
		size_t nevents = ARRAY_LENGTH(events);

		t = &tests[i];
		success = parse_evcode_property(t->prop, events, &nevents);
		litest_assert(success == t->success);
		if (!success)
			continue;

		litest_assert_int_eq(nevents, t->nevents);
		for (size_t j = 0; j < nevents; j++) {
			unsigned int type = events[j].type;
			unsigned int code = events[j].code;
			int value = events[j].value;
			litest_assert_int_eq(t->events[j].type, type);
			litest_assert_int_eq(t->events[j].code, code);
			litest_assert_int_eq(t->events[j].value, value);
		}
	}
}
END_TEST

START_TEST(input_prop_parser)
{
	struct parser_test_val {
		const char *prop;
		bool success;
		size_t nvals;
		struct input_prop values[20];
	} tests[] = {
		{ "+INPUT_PROP_BUTTONPAD", true, 1, {{ INPUT_PROP_BUTTONPAD, true }}},
		{ "+INPUT_PROP_BUTTONPAD;-INPUT_PROP_POINTER", true, 2,
			{ { INPUT_PROP_BUTTONPAD, true },
			  { INPUT_PROP_POINTER, false }}},
		{ "+INPUT_PROP_BUTTONPAD;-0x00;+0x03", true, 3,
			{ { INPUT_PROP_BUTTONPAD, true },
			  { INPUT_PROP_POINTER, false },
			  { INPUT_PROP_SEMI_MT, true }}},
		{ .prop = "", .success = false },
		{ .prop = "0xff", .success = false },
		{ .prop = "INPUT_PROP", .success = false },
		{ .prop = "INPUT_PROP_FOO", .success = false },
		{ .prop = "INPUT_PROP_FOO;INPUT_PROP_FOO", .success = false },
		{ .prop = "INPUT_PROP_POINTER;INPUT_PROP_FOO", .success = false },
		{ .prop = "none", .success = false },
		{ .prop = NULL },
	};
	struct parser_test_val *t;

	for (int i = 0; tests[i].prop; i++) {
		bool success;
		struct input_prop props[32];
		size_t nprops = ARRAY_LENGTH(props);

		t = &tests[i];
		success = parse_input_prop_property(t->prop, props, &nprops);
		litest_assert(success == t->success);
		if (!success)
			continue;

		litest_assert_int_eq(nprops, t->nvals);
		for (size_t j = 0; j < t->nvals; j++) {
			litest_assert_int_eq(t->values[j].prop, props[j].prop);
			litest_assert_int_eq(t->values[j].enabled, props[j].enabled);
		}
	}
}
END_TEST

START_TEST(evdev_abs_parser)
{
	struct test {
		uint32_t which;
		const char *prop;
		int min, max, res, fuzz, flat;

	} tests[] = {
		{ .which = (ABS_MASK_MIN|ABS_MASK_MAX),
		  .prop = "1:2",
		  .min = 1, .max = 2 },
		{ .which = (ABS_MASK_MIN|ABS_MASK_MAX),
		  .prop = "1:2:",
		  .min = 1, .max = 2 },
		{ .which = (ABS_MASK_MIN|ABS_MASK_MAX|ABS_MASK_RES),
		  .prop = "10:20:30",
		  .min = 10, .max = 20, .res = 30 },
		{ .which = (ABS_MASK_RES),
		  .prop = "::100",
		  .res = 100 },
		{ .which = (ABS_MASK_MIN),
		  .prop = "10:",
		  .min = 10 },
		{ .which = (ABS_MASK_MAX|ABS_MASK_RES),
		  .prop = ":10:1001",
		  .max = 10, .res = 1001 },
		{ .which = (ABS_MASK_MIN|ABS_MASK_MAX|ABS_MASK_RES|ABS_MASK_FUZZ),
		  .prop = "1:2:3:4",
		  .min = 1, .max = 2, .res = 3, .fuzz = 4},
		{ .which = (ABS_MASK_MIN|ABS_MASK_MAX|ABS_MASK_RES|ABS_MASK_FUZZ|ABS_MASK_FLAT),
		  .prop = "1:2:3:4:5",
		  .min = 1, .max = 2, .res = 3, .fuzz = 4, .flat = 5},
		{ .which = (ABS_MASK_MIN|ABS_MASK_RES|ABS_MASK_FUZZ|ABS_MASK_FLAT),
		  .prop = "1::3:4:50",
		  .min = 1, .res = 3, .fuzz = 4, .flat = 50},
		{ .which = ABS_MASK_FUZZ|ABS_MASK_FLAT,
		  .prop = ":::5:60",
		  .fuzz = 5, .flat = 60},
		{ .which = ABS_MASK_FUZZ,
		  .prop = ":::5:",
		  .fuzz = 5 },
		{ .which = ABS_MASK_RES, .prop = "::12::",
		  .res = 12 },
		/* Malformed property but parsing this one makes us more
		 * future proof */
		{ .which = (ABS_MASK_RES|ABS_MASK_FUZZ|ABS_MASK_FLAT),
		  .prop = "::12:1:2:3:4:5:6",
		  .res = 12, .fuzz = 1, .flat = 2 },
		{ .which = 0, .prop = ":::::" },
		{ .which = 0, .prop = ":" },
		{ .which = 0, .prop = "" },
		{ .which = 0, .prop = ":asb::::" },
		{ .which = 0, .prop = "foo" },
	};

	ARRAY_FOR_EACH(tests, t) {
		struct input_absinfo abs;
		uint32_t mask;

		mask = parse_evdev_abs_prop(t->prop, &abs);
		litest_assert_int_eq(mask, t->which);

		if (t->which & ABS_MASK_MIN)
			litest_assert_int_eq(abs.minimum, t->min);
		if (t->which & ABS_MASK_MAX)
			litest_assert_int_eq(abs.maximum, t->max);
		if (t->which & ABS_MASK_RES)
			litest_assert_int_eq(abs.resolution, t->res);
		if (t->which & ABS_MASK_FUZZ)
			litest_assert_int_eq(abs.fuzz, t->fuzz);
		if (t->which & ABS_MASK_FLAT)
			litest_assert_int_eq(abs.flat, t->flat);
	}
}
END_TEST

START_TEST(time_conversion)
{
	litest_assert_int_eq(us(10), 10U);
	litest_assert_int_eq(ns2us(10000), 10U);
	litest_assert_int_eq(ms2us(10), 10000U);
	litest_assert_int_eq(s2us(1), 1000000U);
	litest_assert_int_eq(h2us(2), s2us(2 * 60 * 60));
	litest_assert_int_eq(us2ms(10000), 10U);
}
END_TEST

START_TEST(human_time)
{
	struct ht_tests {
		uint64_t interval;
		unsigned int value;
		const char *unit;
	} tests[] = {
		{ 0, 0, "us" },
		{ 123, 123, "us" },
		{ ms2us(5), 5, "ms" },
		{ ms2us(100), 100, "ms" },
		{ s2us(5), 5, "s" },
		{ s2us(100), 100, "s" },
		{ s2us(120), 2, "min" },
		{ 5 * s2us(60), 5, "min" },
		{ 120 * s2us(60), 2, "h" },
		{ 5 * 60 * s2us(60), 5, "h" },
		{ 48 * 60 * s2us(60), 2, "d" },
		{ 1000 * 24 * 60 * s2us(60), 1000, "d" },
		{ 0, 0, NULL },
	};
	for (int i = 0; tests[i].unit != NULL; i++) {
		struct human_time ht;

		ht = to_human_time(tests[i].interval);
		litest_assert_int_eq(ht.value, tests[i].value);
		litest_assert_str_eq(ht.unit, tests[i].unit);
	}
}
END_TEST

struct atoi_test {
	char *str;
	bool success;
	int val;
};

START_TEST(safe_atoi_test)
{
	struct atoi_test tests[] = {
		{ "10", true, 10 },
		{ "20", true, 20 },
		{ "-1", true, -1 },
		{ "2147483647", true, 2147483647 },
		{ "-2147483648", true, -2147483648 },
		{ "4294967295", false, 0 },
		{ "0x0", false, 0 },
		{ "-10x10", false, 0 },
		{ "1x-99", false, 0 },
		{ "", false, 0 },
		{ "abd", false, 0 },
		{ "xabd", false, 0 },
		{ "0xaf", false, 0 },
		{ "0x0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};
	int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atoi(tests[i].str, &v);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xad);
	}
}
END_TEST

START_TEST(safe_atoi_base_16_test)
{
	struct atoi_test tests[] = {
		{ "10", true, 0x10 },
		{ "20", true, 0x20 },
		{ "-1", true, -1 },
		{ "0x10", true, 0x10 },
		{ "0xff", true, 0xff },
		{ "abc", true, 0xabc },
		{ "-10", true, -0x10 },
		{ "0x0", true, 0 },
		{ "0", true, 0 },
		{ "0x-99", false, 0 },
		{ "0xak", false, 0 },
		{ "0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};

	int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atoi_base(tests[i].str, &v, 16);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xad);
	}
}
END_TEST

START_TEST(safe_atoi_base_8_test)
{
	struct atoi_test tests[] = {
		{ "7", true, 07 },
		{ "10", true, 010 },
		{ "20", true, 020 },
		{ "-1", true, -1 },
		{ "010", true, 010 },
		{ "0ff", false, 0 },
		{ "abc", false, 0},
		{ "0xabc", false, 0},
		{ "-10", true, -010 },
		{ "0", true, 0 },
		{ "00", true, 0 },
		{ "0x0", false, 0 },
		{ "0x-99", false, 0 },
		{ "0xak", false, 0 },
		{ "0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};

	int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atoi_base(tests[i].str, &v, 8);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xad);
	}
}
END_TEST

struct atou_test {
	char *str;
	bool success;
	unsigned int val;
};

START_TEST(safe_atou_test)
{
	struct atou_test tests[] = {
		{ "10", true, 10 },
		{ "20", true, 20 },
		{ "-1", false, 0 },
		{ "2147483647", true, 2147483647 },
		{ "-2147483648", false, 0},
		{ "0x0", false, 0 },
		{ "-10x10", false, 0 },
		{ "1x-99", false, 0 },
		{ "", false, 0 },
		{ "abd", false, 0 },
		{ "xabd", false, 0 },
		{ "0xaf", false, 0 },
		{ "0x0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};
	unsigned int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atou(tests[i].str, &v);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xadU);
	}
}
END_TEST

START_TEST(safe_atou_base_16_test)
{
	struct atou_test tests[] = {
		{ "10", true, 0x10 },
		{ "20", true, 0x20 },
		{ "-1", false, 0 },
		{ "0x10", true, 0x10 },
		{ "0xff", true, 0xff },
		{ "abc", true, 0xabc },
		{ "-10", false, 0 },
		{ "0x0", true, 0 },
		{ "0", true, 0 },
		{ "0x-99", false, 0 },
		{ "0xak", false, 0 },
		{ "0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};

	unsigned int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atou_base(tests[i].str, &v, 16);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xadU);
	}
}
END_TEST

START_TEST(safe_atou_base_8_test)
{
	struct atou_test tests[] = {
		{ "7", true, 07 },
		{ "10", true, 010 },
		{ "20", true, 020 },
		{ "-1", false, 0 },
		{ "010", true, 010 },
		{ "0ff", false, 0 },
		{ "abc", false, 0},
		{ "0xabc", false, 0},
		{ "-10", false, 0 },
		{ "0", true, 0 },
		{ "00", true, 0 },
		{ "0x0", false, 0 },
		{ "0x-99", false, 0 },
		{ "0xak", false, 0 },
		{ "0x", false, 0 },
		{ "x10", false, 0 },
		{ NULL, false, 0 }
	};

	unsigned int v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atou_base(tests[i].str, &v, 8);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_int_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xadU);
	}
}
END_TEST

START_TEST(safe_atod_test)
{
	struct atod_test {
		char *str;
		bool success;
		double val;
	} tests[] = {
		{ "10", true, 10 },
		{ "20", true, 20 },
		{ "-1", true, -1 },
		{ "2147483647", true, 2147483647 },
		{ "-2147483648", true, -2147483648 },
		{ "4294967295", true, 4294967295 },
		{ "0x0", false, 0 },
		{ "0x10", false, 0 },
		{ "0xaf", false, 0 },
		{ "x80", false, 0 },
		{ "0.0", true, 0.0 },
		{ "0.1", true, 0.1 },
		{ "1.2", true, 1.2 },
		{ "-324.9", true, -324.9 },
		{ "9324.9", true, 9324.9 },
		{ "NAN", false, 0 },
		{ "INFINITY", false, 0 },
		{ "-10x10", false, 0 },
		{ "1x-99", false, 0 },
		{ "", false, 0 },
		{ "abd", false, 0 },
		{ "xabd", false, 0 },
		{ "0x0x", false, 0 },
		{ NULL, false, 0 }
	};
	double v;
	bool success;

	for (int i = 0; tests[i].str != NULL; i++) {
		v = 0xad;
		success = safe_atod(tests[i].str, &v);
		litest_assert(success == tests[i].success);
		if (success)
			litest_assert_double_eq(v, tests[i].val);
		else
			litest_assert_int_eq(v, 0xad);
	}
}
END_TEST

START_TEST(strsplit_test)
{
	struct strsplit_test {
		const char *string;
		const char *delim;
		const char *results[10];
		const size_t nresults;
	} tests[] = {
		{ "one two three", " ", { "one", "two", "three", NULL }, 3 },
		{ "one two\tthree", " \t", { "one", "two", "three", NULL }, 3 },
		{ "one", " ", { "one", NULL }, 1 },
		{ "one two ", " ", { "one", "two", NULL }, 2 },
		{ "one  two", " ", { "one", "two", NULL }, 2 },
		{ " one two", " ", { "one", "two", NULL }, 2 },
		{ "one", "\t \r", { "one", NULL }, 1 },
		{ "one two three", " t", { "one", "wo", "hree", NULL }, 3 },
		{ " one two three", "te", { " on", " ", "wo ", "hr", NULL }, 4 },
		{ "one", "ne", { "o", NULL }, 1 },
		{ "onene", "ne", { "o", NULL }, 1 },
		{ "+1-2++3--4++-+5-+-", "+-", { "1", "2", "3", "4", "5", NULL }, 5 },
		/* special cases */
		{ "", " ", { NULL }, 0 },
		{ " ", " ", { NULL }, 0 },
		{ "     ", " ", { NULL }, 0 },
		{ "oneoneone", "one", { NULL} , 0 },
		{ NULL, NULL, { NULL }, 0}
	};
	struct strsplit_test *t = tests;

	while (t->string) {
		size_t nelem;
		char **strv = strv_from_string(t->string, t->delim, &nelem);

		for (size_t idx = 0; idx < t->nresults; idx++)
			litest_assert_str_eq(t->results[idx], strv[idx]);

		litest_assert_int_eq(nelem, t->nresults);

		/* When there are no elements validate return value is Null,
		   otherwise validate result array is Null terminated. */
		if(t->nresults == 0)
			litest_assert_ptr_eq(strv, NULL);
		else
			litest_assert_ptr_eq(strv[t->nresults], NULL);

		strv_free(strv);
		t++;
	}
}
END_TEST

struct strv_test_data {
	const char *terminate_at;
	unsigned char bitmask[1];
};

static int strv_test_set_bitmask(const char *str, size_t index, void *data)
{
	struct strv_test_data *td = data;

	if (streq(str, td->terminate_at))
		return index + 1;

	set_bit(td->bitmask, index);

	return 0;
}

START_TEST(strv_for_each_test)
{
	struct test_data {
		const char *terminator;
		int index;
		unsigned int bitmask;
	} test_data[] = {
		{ "one", 1, 0x0 },
		{ "two", 2, 0x1 },
		{ "three", 3, 0x3 },
		{ "four", 4, 0x7 },
		{ "five", 5, 0xf },
		{ "does-not-exist", 0, 0x1f },
		{ NULL, 0, 0x1f },
		{ NULL, 0 },
	};
	const char *array[] = { "one", "two", "three", "four", "five", NULL };
	struct test_data *t = test_data;

	while (t->terminator || t->bitmask) {
		const int max = 3;
		struct strv_test_data td = {
			.terminate_at = t->terminator,
			.bitmask = { 0 },
		};

		int rc = strv_for_each(array, strv_test_set_bitmask, &td);
		litest_assert_int_eq(rc, t->index);
		litest_assert_int_eq(td.bitmask[0], t->bitmask);

		struct strv_test_data tdmax = {
			.terminate_at = t->terminator,
			.bitmask = { 0 },
		};

		rc = strv_for_each_n(array, max, strv_test_set_bitmask, &tdmax);
		if (max < t->index)
			litest_assert_int_eq(rc, 0);
		else
			litest_assert_int_eq(rc, t->index);
		litest_assert_int_eq(tdmax.bitmask[0], t->bitmask & ((1 << max) - 1));

		t++;
	}
}
END_TEST

START_TEST(double_array_from_string_test)
{
	struct double_array_from_string_test {
		const char *string;
		const char *delim;
		const double array[10];
		const size_t len;
		const bool result;
	} tests[] = {
		{ "1 2 3", " ", { 1, 2, 3 }, 3 },
		{ "1", " ", { 1 }, 1 },
		{ "1,2.5,", ",", { 1, 2.5 }, 2 },
		{ "1.0  2", " ", { 1, 2.0 }, 2 },
		{ " 1 2", " ", { 1, 2 }, 2 },
		{ " ; 1;2  3.5  ;;4.1", "; ", { 1, 2, 3.5, 4.1 }, 4 },
		/* special cases */
		{ "1 two", " ", { 0 }, 0 },
		{ "one two", " ", { 0 }, 0 },
		{ "one 2", " ", { 0 }, 0 },
		{ "", " ", { 0 }, 0 },
		{ " ", " ", { 0 }, 0 },
		{ "    ", " ", { 0 }, 0 },
		{ "", " ", { 0 }, 0 },
		{ "oneoneone", "one", { 0 }, 0 },
		{ NULL, NULL, { 0 }, 0 }
	};
	struct double_array_from_string_test *t = tests;

	while (t->string) {
		size_t len;
		double *array = double_array_from_string(t->string,
							 t->delim,
							 &len);
		litest_assert_int_eq(len, t->len);

		for (size_t idx = 0; idx < len; idx++) {
			litest_assert_ptr_notnull(array);
			litest_assert_double_eq(array[idx], t->array[idx]);
		}

		free(array);
		t++;
	}
}
END_TEST

START_TEST(strargv_test)
{
	struct argv_test {
		int argc;
		char *argv[10];
		int expected;
	} tests[] = {
		{ 0, {NULL}, 0 },
		{ 1, {"hello", "World"}, 1 },
		{ 2, {"hello", "World"}, 2 },
		{ 2, {"", " "}, 2 },
		{ 2, {"", NULL}, 0 },
		{ 2, {NULL, NULL}, 0 },
		{ 1, {NULL, NULL}, 0 },
		{ 3, {"hello", NULL, "World"}, 0 },
	};

	ARRAY_FOR_EACH(tests, t) {
		char **strv = strv_from_argv(t->argc, t->argv);

		if (t->expected == 0) {
			litest_assert(strv == NULL);
		} else {
			int count = 0;
			char **s = strv;
			while (*s) {
				litest_assert_str_eq(*s, t->argv[count]);
				count++;
				s++;
			}
			litest_assert_int_eq(t->expected, count);
			strv_free(strv);
		}
	}
}
END_TEST

START_TEST(kvsplit_double_test)
{
	struct kvsplit_dbl_test {
		const char *string;
		const char *psep;
		const char *kvsep;
		ssize_t nresults;
		struct {
			double a;
			double b;
		} results[32];
	} tests[] = {
		{ "1:2;3:4;5:6", ";", ":", 3, { {1, 2}, {3, 4}, {5, 6}}},
		{ "1.0x2.3 -3.2x4.5 8.090909x-6.00", " ", "x", 3, { {1.0, 2.3}, {-3.2, 4.5}, {8.090909, -6}}},

		{ "1:2", "x", ":", 1, {{1, 2}}},
		{ "1:2", ":", "x", -1, {}},
		{ "1:2", NULL, "x", -1, {}},
		{ "1:2", "", "x", -1, {}},
		{ "1:2", "x", NULL, -1, {}},
		{ "1:2", "x", "", -1, {}},
		{ "a:b", "x", ":", -1, {}},
		{ "", " ", "x", -1, {}},
		{ "1.2.3.4.5", ".", "", -1, {}},
		{ NULL }
	};
	struct kvsplit_dbl_test *t = tests;

	while (t->string) {
		struct key_value_double *result = NULL;
		ssize_t npairs;

		npairs = kv_double_from_string(t->string,
					       t->psep,
					       t->kvsep,
					       &result);
		litest_assert_int_eq(npairs, t->nresults);

		for (ssize_t i = 0; i < npairs; i++) {
			litest_assert_double_eq(t->results[i].a, result[i].key);
			litest_assert_double_eq(t->results[i].b, result[i].value);
		}

		free(result);
		t++;
	}
}
END_TEST

START_TEST(strjoin_test)
{
	struct strjoin_test {
		char *strv[10];
		const char *joiner;
		const char *result;
	} tests[] = {
		{ { "one", "two", "three", NULL }, " ", "one two three" },
		{ { "one", NULL }, "x", "one" },
		{ { "one", "two", NULL }, "x", "onextwo" },
		{ { "one", "two", NULL }, ",", "one,two" },
		{ { "one", "two", NULL }, ", ", "one, two" },
		{ { "one", "two", NULL }, "one", "oneonetwo" },
		{ { "one", "two", NULL }, NULL, NULL },
		{ { "", "", "", NULL }, " ", "  " },
		{ { "a", "b", "c", NULL }, "", "abc" },
		{ { "", "b", "c", NULL }, "x", "xbxc" },
		{ { "", "", "", NULL }, "", "" },
		{ { NULL }, NULL, NULL }
	};
	struct strjoin_test *t = tests;
	struct strjoin_test nulltest = { {NULL}, "x", NULL };

	while (t->strv[0]) {
		char *str;
		str = strv_join(t->strv, t->joiner);
		if (t->result == NULL)
			litest_assert(str == NULL);
		else
			litest_assert_str_eq(str, t->result);
		free(str);
		t++;
	}

	litest_assert(strv_join(nulltest.strv, "x") == NULL);
}
END_TEST

START_TEST(strstrip_test)
{
	struct strstrip_test {
		const char *string;
		const char *expected;
		const char *what;
	} tests[] = {
		{ "foo",		"foo",		"1234" },
		{ "\"bar\"",		"bar",		"\"" },
		{ "'bar'",		"bar",		"'" },
		{ "\"bar\"",		"\"bar\"",	"'" },
		{ "'bar'",		"'bar'",	"\"" },
		{ "\"bar\"",		"bar",		"\"" },
		{ "\"\"",		"",		"\"" },
		{ "\"foo\"bar\"",	"foo\"bar",	"\"" },
		{ "\"'foo\"bar\"",	"foo\"bar",	"\"'" },
		{ "abcfooabcbarbca",	"fooabcbar",	"abc" },
		{ "xxxxfoo",		"foo",		"x" },
		{ "fooyyyy",		"foo",		"y" },
		{ "xxxxfooyyyy",	"foo",		"xy" },
		{ "x xfooy y",		" xfooy ",	"xy" },
		{ " foo\n",		"foo",		" \n" },
		{ "",			"",		"abc" },
		{ "",			"",		"" },
		{ NULL , NULL, NULL }
	};
	struct strstrip_test *t = tests;

	while (t->string) {
		char *str;
		str = strstrip(t->string, t->what);
		litest_assert_str_eq(str, t->expected);
		free(str);
		t++;
	}
}
END_TEST

START_TEST(strendswith_test)
{
	struct strendswith_test {
		const char *string;
		const char *suffix;
		bool expected;
	} tests[] = {
		{ "foobar", "bar", true },
		{ "foobar", "foo", false },
		{ "foobar", "foobar", true },
		{ "foo", "foobar", false },
		{ "foobar", "", false },
		{ "", "", false },
		{ "", "foo", false },
		{ NULL, NULL, false },
	};

	for (struct strendswith_test *t = tests; t->string; t++) {
		litest_assert_int_eq(strendswith(t->string, t->suffix),
				 t->expected);
	}
}
END_TEST

START_TEST(strstartswith_test)
{
	struct strstartswith_test {
		const char *string;
		const char *suffix;
		bool expected;
	} tests[] = {
		{ "foobar", "foo", true },
		{ "foobar", "bar", false },
		{ "foobar", "foobar", true },
		{ "foo", "foobar", false },
		{ "foo", "", false },
		{ "", "", false },
		{ "foo", "", false },
		{ NULL, NULL, false },
	};

	for (struct strstartswith_test *t = tests; t->string; t++) {
		litest_assert_int_eq(strstartswith(t->string, t->suffix),
				 t->expected);
	}
}
END_TEST

START_TEST(strsanitize_test)
{
	struct strsanitize_test {
		const char *string;
		const char *expected;
	} tests[] = {
		{ "foobar", "foobar" },
		{ "", "" },
		{ "%", "%%" },
		{ "%%%%", "%%%%%%%%" },
		{ "x %s", "x %%s" },
		{ "x %", "x %%" },
		{ "%sx", "%%sx" },
		{ "%s%s", "%%s%%s" },
		{ NULL, NULL },
	};

	for (struct strsanitize_test *t = tests; t->string; t++) {
		char *sanitized = str_sanitize(t->string);
		litest_assert_str_eq(sanitized, t->expected);
		free(sanitized);
	}
}
END_TEST

START_TEST(list_test_insert)
{
	struct list_test {
		int val;
		struct list node;
	} tests[] = {
		{ .val  = 1 },
		{ .val  = 2 },
		{ .val  = 3 },
		{ .val  = 4 },
	};
	struct list_test *t;
	struct list head;
	int val;

	list_init(&head);

	ARRAY_FOR_EACH(tests, t) {
		list_insert(&head, &t->node);
	}

	val = 4;
	list_for_each(t, &head, node) {
		litest_assert_int_eq(t->val, val);
		val--;
	}

	litest_assert_int_eq(val, 0);
}
END_TEST

START_TEST(list_test_append)
{
	struct list_test {
		int val;
		struct list node;
	} tests[] = {
		{ .val  = 1 },
		{ .val  = 2 },
		{ .val  = 3 },
		{ .val  = 4 },
	};
	struct list_test *t;
	struct list head;
	int val;

	list_init(&head);

	ARRAY_FOR_EACH(tests, t) {
		list_append(&head, &t->node);
	}

	val = 1;
	list_for_each(t, &head, node) {
		litest_assert_int_eq(t->val, val);
		val++;
	}
	litest_assert_int_eq(val, 5);
}
END_TEST

START_TEST(list_test_foreach)
{
	struct list_test {
		int val;
		struct list node;
	} tests[] = {
		{ .val  = 1 },
		{ .val  = 2 },
		{ .val  = 3 },
		{ .val  = 4 },
	};
	struct list_test *t;
	struct list head;

	list_init(&head);

	ARRAY_FOR_EACH(tests, t) {
		list_append(&head, &t->node);
	}

	/* Make sure both loop macros are a single line statement */
	if (false)
		list_for_each(t, &head, node) {
			litest_abort_msg("We should not get here");
		}

	if (false)
		list_for_each_safe(t, &head, node) {
			litest_abort_msg("We should not get here");
		}
}
END_TEST

START_TEST(strverscmp_test)
{
	litest_assert_int_eq(libinput_strverscmp("", ""), 0);
	litest_assert_int_gt(libinput_strverscmp("0.0.1", ""), 0);
	litest_assert_int_lt(libinput_strverscmp("", "0.0.1"), 0);
	litest_assert_int_eq(libinput_strverscmp("0.0.1", "0.0.1"), 0);
	litest_assert_int_eq(libinput_strverscmp("0.0.1", "0.0.2"), -1);
	litest_assert_int_eq(libinput_strverscmp("0.0.2", "0.0.1"), 1);
	litest_assert_int_eq(libinput_strverscmp("0.0.1", "0.1.0"), -1);
	litest_assert_int_eq(libinput_strverscmp("0.1.0", "0.0.1"), 1);
}
END_TEST

START_TEST(streq_test)
{
	litest_assert(streq("", "") == true);
	litest_assert(streq(NULL, NULL) == true);
	litest_assert(streq("0.0.1", "") == false);
	litest_assert(streq("foo", NULL) == false);
	litest_assert(streq(NULL, "foo") == false);
	litest_assert(streq("0.0.1", "0.0.1") == true);
}
END_TEST

START_TEST(strneq_test)
{
	litest_assert(strneq("", "", 1) == true);
	litest_assert(strneq(NULL, NULL, 1) == true);
	litest_assert(strneq("0.0.1", "", 6) == false);
	litest_assert(strneq("foo", NULL, 5) == false);
	litest_assert(strneq(NULL, "foo", 5) == false);
	litest_assert(strneq("0.0.1", "0.0.1", 6) == true);
}
END_TEST

START_TEST(basename_test)
{
	struct test {
		const char *path;
		const char *expected;
	} tests[] = {
		{ "a", "a" },
		{ "foo.c", "foo.c" },
		{ "foo", "foo" },
		{ "/path/to/foo.h", "foo.h" },
		{ "../bar.foo", "bar.foo" },
		{ "./bar.foo.baz", "bar.foo.baz" },
		{ "./", NULL },
		{ "/", NULL },
		{ "/bar/", NULL },
		{ "/bar", "bar" },
		{ "", NULL },
	};

	ARRAY_FOR_EACH(tests, t) {
		const char *result = safe_basename(t->path);
		if (t->expected == NULL)
			litest_assert(result == NULL);
		else
			litest_assert_str_eq(result, t->expected);
	}
}
END_TEST

START_TEST(trunkname_test)
{
	struct test {
		const char *path;
		const char *expected;
	} tests[] = {
		{ "foo.c", "foo" },
		{ "/path/to/foo.h", "foo" },
		{ "/path/to/foo", "foo" },
		{ "../bar.foo", "bar" },
		{ "./bar.foo.baz", "bar.foo" },
		{ "./", "" },
		{ "/", "" },
		{ "/bar/", "" },
		{ "/bar", "bar" },
		{ "", "" },
	};

	ARRAY_FOR_EACH(tests, t) {
		char *result = trunkname(t->path);
		litest_assert_str_eq(result, t->expected);
		free(result);
	}
}
END_TEST

START_TEST(absinfo_normalize_value_test)
{
	struct input_absinfo abs = {
		.minimum = 0,
		.maximum = 100,
	};

	litest_assert_double_eq(absinfo_normalize_value(&abs, -100), 0.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, -1), 0.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 0), 0.0);
	litest_assert_double_gt(absinfo_normalize_value(&abs, 1), 0.0);
	litest_assert_double_lt(absinfo_normalize_value(&abs, 99), 1.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 100), 1.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 101), 1.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 200), 1.0);

	abs.minimum = -50;
	abs.maximum = 50;

	litest_assert_double_eq(absinfo_normalize_value(&abs, -51), 0.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, -50), 0.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 0), 0.5);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 50), 1.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 51), 1.0);

	abs.minimum = -50;
	abs.maximum = 0;

	litest_assert_double_eq(absinfo_normalize_value(&abs, -51), 0.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, -50), 0.0);
	litest_assert_double_gt(absinfo_normalize_value(&abs, -49), 0.0);
	litest_assert_double_lt(absinfo_normalize_value(&abs, -1), 1.0);
	litest_assert_double_eq(absinfo_normalize_value(&abs, 0), 1.0);
}
END_TEST

START_TEST(range_test)
{
	struct range incl = range_init_inclusive(1, 100);
	litest_assert_int_eq(incl.lower, 1);
	litest_assert_int_eq(incl.upper, 101);

	struct range excl = range_init_exclusive(1, 100);
	litest_assert_int_eq(excl.lower, 1);
	litest_assert_int_eq(excl.upper, 100);

	struct range zero = range_init_exclusive(0, 0);
	litest_assert_int_eq(zero.lower, 0);
	litest_assert_int_eq(zero.upper, 0);

	struct range empty = range_init_empty();
	litest_assert_int_eq(empty.lower, 0);
	litest_assert_int_eq(empty.upper, -1);

	litest_assert(range_is_valid(&incl));
	litest_assert(range_is_valid(&excl));
	litest_assert(!range_is_valid(&zero));
	litest_assert(!range_is_valid(&empty));

	int expected = 1;
	int r = 0;
	range_for_each(&incl, r) {
		litest_assert_int_eq(r, expected);
		expected++;
	}
	litest_assert_int_eq(r, 101);

}
END_TEST

START_TEST(stringbuf_test)
{
	struct stringbuf buf;
	struct stringbuf *b = &buf;
	int rc;

	stringbuf_init(b);
	litest_assert_int_eq(b->len, 0u);

	rc = stringbuf_append_string(b, "foo");
	litest_assert_neg_errno_success(rc);
	rc = stringbuf_append_string(b, "bar");
	litest_assert_neg_errno_success(rc);
	rc = stringbuf_append_string(b, "baz");
	litest_assert_neg_errno_success(rc);
	litest_assert_str_eq(b->data, "foobarbaz");
	litest_assert_int_eq(b->len, strlen("foobarbaz"));

	rc = stringbuf_ensure_space(b, 500);
	litest_assert_neg_errno_success(rc);
	litest_assert_int_ge(b->sz, 500u);

	rc = stringbuf_ensure_size(b, 0);
	litest_assert_neg_errno_success(rc);
	rc = stringbuf_ensure_size(b, 1024);
	litest_assert_neg_errno_success(rc);
	litest_assert_int_ge(b->sz, 1024u);

	char *data = stringbuf_steal(b);
	litest_assert_str_eq(data, "foobarbaz");
	litest_assert_int_eq(b->sz, 0u);
	litest_assert_int_eq(b->len, 0u);
	litest_assert_ptr_null(b->data);
	free(data);

	const char *str = "1234567890";
	rc = stringbuf_append_string(b, str);
	litest_assert_neg_errno_success(rc);
	litest_assert_str_eq(b->data, str);
	litest_assert_int_eq(b->len, 10u);
	stringbuf_reset(b);

	/* intentional double-reset */
	stringbuf_reset(b);

	int pipefd[2];
	rc = pipe2(pipefd, O_CLOEXEC | O_NONBLOCK);
	litest_assert_neg_errno_success(rc);

	str = "foo bar baz";
	char *compare = NULL;
	for (int i = 0; i < 100; i++) {
		rc = write(pipefd[1], str, strlen(str));
		litest_assert_neg_errno_success(rc);

		rc = stringbuf_append_from_fd(b, pipefd[0], 64);
		litest_assert_neg_errno_success(rc);

		char *expected;
		rc = xasprintf(&expected, "%s%s", compare ? compare : "", str);
		litest_assert_neg_errno_success(rc);
		litest_assert_str_eq(b->data, expected);

		free(compare);
		compare = expected;
	}
	free(compare);
	close(pipefd[0]);
	close(pipefd[1]);
	stringbuf_reset(b);

	rc = pipe2(pipefd, O_CLOEXEC | O_NONBLOCK);
	litest_assert_neg_errno_success(rc);

	const size_t stride = 256;
	const size_t maxsize = 4096;

	for (size_t i = 0; i < maxsize; i += stride) {
		char buf[stride];
		memset(buf, i/stride, sizeof(buf));
		rc = write(pipefd[1], buf, sizeof(buf));
		litest_assert_neg_errno_success(rc);
	}

	stringbuf_append_from_fd(b, pipefd[0], 0);
	litest_assert_int_eq(b->len, maxsize);
	litest_assert_int_ge(b->sz, maxsize);

	for (size_t i = 0; i < maxsize; i+= stride) {
		char buf[stride];
		memset(buf, i/stride, sizeof(buf));
		litest_assert_int_eq(memcmp(buf, b->data + i, sizeof(buf)), 0);
	}

	close(pipefd[0]);
	close(pipefd[1]);
	stringbuf_reset(b);
}
END_TEST

START_TEST(multivalue_test)
{
	{
		struct multivalue v = multivalue_new_string("test");
		litest_assert_int_eq(v.type, 's');
		litest_assert_str_eq(v.value.s, "test");

		const char *s;
		multivalue_extract_typed(&v, 's', &s);
		litest_assert_str_eq(s, "test");
		litest_assert_ptr_eq(s, (const char*)v.value.s);
		multivalue_extract(&v, &s);
		litest_assert_str_eq(s, "test");
		litest_assert_ptr_eq(s, (const char*)v.value.s);

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_str_eq(copy.value.s, v.value.s);
		char *p1 = copy.value.s;
		char *p2 = v.value.s;
		litest_assert_ptr_ne(p1, p2);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "test");
		free(str);
	}

	{
		struct multivalue v = multivalue_new_char('x');
		litest_assert_int_eq(v.type, 'c');
		litest_assert_int_eq(v.value.c, 'x');

		char c;
		multivalue_extract_typed(&v, 'c', &c);
		litest_assert_int_eq(c, 'x');
		multivalue_extract(&v, &c);
		litest_assert_int_eq(c, 'x');

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_int_eq(copy.value.c, v.value.c);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "x");
		free(str);
	}

	{
		struct multivalue v = multivalue_new_u32(0x1234);
		litest_assert_int_eq(v.type, 'u');
		litest_assert_int_eq(v.value.u, 0x1234u);

		uint32_t c;
		multivalue_extract_typed(&v, 'u', &c);
		litest_assert_int_eq(c, 0x1234u);
		multivalue_extract(&v, &c);
		litest_assert_int_eq(c, 0x1234u);

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_int_eq(copy.value.u, v.value.u);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "4660");
		free(str);
	}

	{
		struct multivalue v = multivalue_new_i32(-123);
		litest_assert_int_eq(v.type, 'i');
		litest_assert_int_eq(v.value.i, -123);

		int32_t c;
		multivalue_extract_typed(&v, 'i', &c);
		litest_assert_int_eq(c, -123);
		multivalue_extract(&v, &c);
		litest_assert_int_eq(c, -123);

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_int_eq(copy.value.i, v.value.i);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "-123");
		free(str);
	}

	{
		struct multivalue v = multivalue_new_bool(true);
		litest_assert_int_eq(v.type, 'b');
		litest_assert_int_eq(v.value.b, true);

		bool c;
		multivalue_extract_typed(&v, 'b', &c);
		litest_assert_int_eq(c, true);
		multivalue_extract(&v, &c);
		litest_assert_int_eq(c, true);

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_int_eq(copy.value.b, v.value.b);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "true");
		free(str);
	}

	{
		struct multivalue v = multivalue_new_double(0.1234);
		litest_assert_int_eq(v.type, 'd');
		litest_assert_double_eq(v.value.d, 0.1234);

		double c;
		multivalue_extract_typed(&v, 'd', &c);
		litest_assert_double_eq(c, 0.1234);
		multivalue_extract(&v, &c);
		litest_assert_double_eq(c, 0.1234);

		struct multivalue copy = multivalue_copy(&v);
		litest_assert_int_eq(copy.type, v.type);
		litest_assert_double_eq(copy.value.d, v.value.d);

		char *str = multivalue_as_str(&v);
		litest_assert_str_eq(str, "0.123400");
		free(str);
	}

}
END_TEST

int main(void)
{
	struct litest_runner *runner = litest_runner_new();

	/* not worth forking the tests here */
	litest_runner_set_num_parallel(runner, 0);

#define ADD_TEST(func_) do { \
	struct litest_runner_test_description tdesc =  { \
		.func = func_, \
	};\
	snprintf(tdesc.name, sizeof(tdesc.name), # func_); \
	litest_runner_add_test(runner, &tdesc); \
} while(0)

	ADD_TEST(array_for_each);

	ADD_TEST(bitfield_helpers);
	ADD_TEST(matrix_helpers);
	ADD_TEST(ratelimit_helpers);
	ADD_TEST(dpi_parser);
	ADD_TEST(wheel_click_parser);
	ADD_TEST(wheel_click_count_parser);
	ADD_TEST(dimension_prop_parser);
	ADD_TEST(reliability_prop_parser);
	ADD_TEST(calibration_prop_parser);
	ADD_TEST(range_prop_parser);
	ADD_TEST(boolean_prop_parser);
	ADD_TEST(evcode_prop_parser);
	ADD_TEST(input_prop_parser);
	ADD_TEST(evdev_abs_parser);
	ADD_TEST(safe_atoi_test);
	ADD_TEST(safe_atoi_base_16_test);
	ADD_TEST(safe_atoi_base_8_test);
	ADD_TEST(safe_atou_test);
	ADD_TEST(safe_atou_base_16_test);
	ADD_TEST(safe_atou_base_8_test);
	ADD_TEST(safe_atod_test);
	ADD_TEST(strsplit_test);
	ADD_TEST(strv_for_each_test);
	ADD_TEST(double_array_from_string_test);
	ADD_TEST(strargv_test);
	ADD_TEST(kvsplit_double_test);
	ADD_TEST(strjoin_test);
	ADD_TEST(strstrip_test);
	ADD_TEST(strendswith_test);
	ADD_TEST(strstartswith_test);
	ADD_TEST(strsanitize_test);
	ADD_TEST(time_conversion);
	ADD_TEST(human_time);

	ADD_TEST(list_test_insert);
	ADD_TEST(list_test_append);
	ADD_TEST(list_test_foreach);
	ADD_TEST(strverscmp_test);
	ADD_TEST(streq_test);
	ADD_TEST(strneq_test);
	ADD_TEST(trunkname_test);
	ADD_TEST(basename_test);

	ADD_TEST(absinfo_normalize_value_test);

	ADD_TEST(range_test);
	ADD_TEST(stringbuf_test);
	ADD_TEST(multivalue_test);

	enum litest_runner_result result = litest_runner_run_tests(runner);
	litest_runner_destroy(runner);

	if (result == LITEST_SKIP)
		return 77;

	return result - LITEST_PASS;
}
