/* hexter DSSI software synthesizer GUI
 *
 * Copyright (C) 2011 Sean Bolton.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA.
 */

#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "hexter.h"
#include "gui_retro_console.h"

GtkWidget *rc_drawing_area;

static cairo_surface_t *rc_font_surface[RC_MAX_COLOR_PAIRS];

static int rc_console_width;
static int rc_console_height;
static int rc_cursor_x;
static int rc_cursor_y;
static int rc_color_pair;

static void (*rc_button_press_callback)(guint, guint, guint);
static void (*rc_key_press_callback)(gboolean, guint, guint);

typedef struct _rc_char_t {
    char c, pair;
} rc_char_t;

static rc_char_t *rc_chars;

static struct _colors {
    float r, g, b;
} colors[RC_COLOR_COUNT] = {
    { 1, 1, 1 },  /* white  */
    { 0, 0, 0 },  /* black  */
    { 0, 1, 0 },  /* green  */
    { 0, 0, 1 },  /* blue   */
    { 1, 1, 0 },  /* yellow */
    { 0, 1, 1 }   /* cyan   */
};

/* static rc_char_t *get_rc_char(int x, int y); */
#define get_rc_char(x, y) (rc_chars + (y * rc_console_width) + x)

void
rc_console_set_button_press_callback(void (*cb)(guint, guint, guint))
{
    rc_button_press_callback = cb;
}

static gboolean
rc_button_press (GtkWidget      *widget,
                 GdkEventButton *event,
                 gpointer        user_data)
{
    /* GUIDB_MESSAGE(DB_GUI, " rc_button_press: button %d @ %g, %g\n", event->button, event->x, event->y); */

    if (!GTK_WIDGET_HAS_FOCUS (widget))
	gtk_widget_grab_focus (widget);

    if (rc_button_press_callback)
        (*rc_button_press_callback)(event->button, ((int)event->x) / RC_FONT_WIDTH,
                                                   ((int)event->y) / RC_FONT_HEIGHT);

    return FALSE; /* false to propagate further, true to stop */
}

void
rc_console_set_key_press_callback(void (*cb)(gboolean, guint, guint))
{
    rc_key_press_callback = cb;
}

static guint keyvals_to_pass[] = {
    GDK_BackSpace, GDK_Delete, GDK_Escape, GDK_Return, 
    GDK_Tab, GDK_ISO_Left_Tab,
    GDK_Page_Up, GDK_Page_Down, GDK_Home, GDK_End,
    GDK_Left, GDK_Right, GDK_Up, GDK_Down,
    GDK_F1, GDK_F2, GDK_F3, GDK_F4, GDK_F5, GDK_F6,
    0
};

static gboolean
rc_key_press (GtkWidget   *widget,
              GdkEventKey *event,
              gpointer     user_data)
{
    gboolean pressed;
    guint u, state;

    if (rc_key_press_callback == NULL)
        return FALSE;  /* propagate to other handlers */

    pressed = (event->type == GDK_KEY_PRESS);
    /* "meta" on OS X is Apple is 16 + 2^28
     * "meta" on (my) linux is Alt is 8     */
    state = event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK | GDK_MOD2_MASK);

    for (u = 0; keyvals_to_pass[u]; u++)
        if (event->keyval == keyvals_to_pass[u]) {
            (*rc_key_press_callback)(pressed, event->keyval, state);
            return TRUE;
        }

    u = gdk_keyval_to_unicode(event->keyval);

    /* This is way too restrictive, but we only need what the DX-7 supported. */
    if (u == 0xA5) u =  92; /* yen */
    if (u == 0xAB) u = 127; /* << */
    if (u == 0xBB) u = 126; /* >> */
    if (u >= ' ' && u <= 127) {
        (*rc_key_press_callback)(pressed, u, state);
        return TRUE;
    } else {
        GUIDB_MESSAGE(DB_GUI, " rc_key_press: %s keyval $%02x with state %d, unicode %u\n",
                      pressed ? "pressed" : "released",
                      event->keyval, event->state, gdk_keyval_to_unicode(event->keyval));
    }

    return FALSE; /* false to propagate further, true to stop */
}

void
rc_console_addch(char c)
{
    if (rc_cursor_x < rc_console_width && rc_cursor_y < rc_console_height) {
        rc_char_t *rch = get_rc_char(rc_cursor_x, rc_cursor_y);

        rch->c = c;
        rch->pair = rc_color_pair;
        
        gtk_widget_queue_draw_area(rc_drawing_area, rc_cursor_x * RC_FONT_WIDTH, rc_cursor_y * RC_FONT_HEIGHT,
                                   RC_FONT_WIDTH, RC_FONT_HEIGHT);
    }
    rc_cursor_x++;
}

void
rc_console_addstr(const char *t)
{
    for (; *t; t++) {
        if (*t == '\n') {
            rc_cursor_x = 0;
            rc_cursor_y++;
        } else {
            rc_console_addch(*t);
        }
    }
}

void
rc_console_move(int y, int x)
{
    if (x >= 0 && x < rc_console_width && y >= 0 && y < rc_console_height) {
        rc_cursor_x = x;
        rc_cursor_y = y;
    }
}

void
rc_console_clrtoeol(void)
{
    int x;

    for (x = rc_cursor_x; x < rc_console_width; x++) {
        rc_char_t *rch = get_rc_char(x, rc_cursor_y);

        rch->c = ' ';
        rch->pair = rc_color_pair;
    }

    gtk_widget_queue_draw_area(rc_drawing_area,
                               rc_cursor_x * RC_FONT_WIDTH,
                               rc_cursor_y * RC_FONT_HEIGHT,
                               (rc_console_width - rc_cursor_x) * RC_FONT_WIDTH,
                               RC_FONT_HEIGHT);
}

void
rc_console_erase(void)
{
    for (rc_cursor_y = 0; rc_cursor_y < rc_console_height; rc_cursor_y++) {
        for (rc_cursor_x = 0; rc_cursor_x < rc_console_width; rc_cursor_x++) {
            rc_char_t *rch = get_rc_char(rc_cursor_x, rc_cursor_y);

            rch->c = ' ';
            rch->pair = rc_color_pair;
        }
    }
    rc_cursor_x = rc_cursor_y = 0;

    gtk_widget_queue_draw_area(rc_drawing_area,
                               0, 0,
                               rc_console_width * RC_FONT_WIDTH,
                               rc_console_height * RC_FONT_HEIGHT);
}

void
rc_console_init_color_pair(int pair, int fg, int bg)
{
    int width, height;
    cairo_t *cr;

    if (pair < 1 || pair >= RC_MAX_COLOR_PAIRS || rc_font_surface[pair] != NULL)
        return;

    /* Render the font to a cache surface in the requested foreground and
     * background colors. Doing this once here allows for using simply
     * cairo_set_source_surface(), cairo_rectangle(), cairo_fill() in
     * rc_expose(), which is much faster. */
    width  = cairo_image_surface_get_width(rc_font_surface[0]);
    height = cairo_image_surface_get_height(rc_font_surface[0]);
    rc_font_surface[pair] = cairo_surface_create_similar(rc_font_surface[0],
                                                         CAIRO_CONTENT_COLOR,
                                                         width, height);
    cr = cairo_create(rc_font_surface[pair]);
    cairo_set_source_rgb(cr, colors[bg].r, colors[bg].g, colors[bg].b);
    cairo_paint(cr);
    cairo_set_source_rgb(cr, colors[fg].r, colors[fg].g, colors[fg].b);
    cairo_mask_surface (cr, rc_font_surface[0], 0, 0);
    cairo_destroy(cr);
}

void
rc_console_set_color_pair(int pair)
{
    rc_color_pair = (pair < 0) ? 0 : ((pair >= RC_MAX_COLOR_PAIRS) ? RC_MAX_COLOR_PAIRS - 1 : pair);
}

static gboolean
rc_expose(GtkWidget *widget, GdkEventExpose *event)
{
    cairo_t *cr = gdk_cairo_create (widget->window);
    int x0 = event->area.x / RC_FONT_WIDTH;
    int x1 = (event->area.x + event->area.width - 1) / RC_FONT_WIDTH;
    int y0 = event->area.y / RC_FONT_HEIGHT;
    int y1 = (event->area.y + event->area.height - 1) / RC_FONT_HEIGHT;
    int x, y;

    /* GUIDB_MESSAGE(DB_GUI, " rc_expose: %d x %d @ %d, %d => %d, %d to %d, %d\n",
                     event->area.width, event->area.height, event->area.x, event->area.y,
                     x0, y0, x1, y1); */

    if (x1 >= rc_console_width)  x1 = rc_console_width - 1;
    if (y1 >= rc_console_height) y1 = rc_console_height - 1;

    for (y = y0; y <= y1; y++) {
        for (x = x0; x <= x1; x++) {
            rc_char_t *rch = get_rc_char(x, y);
            unsigned char c = (unsigned char)rch->c;
            int pair = rch->pair;

            if (rc_font_surface[pair] == NULL)
                continue;

            if (c >= 128 && c < 139) c = 128;
            else if (c > 153) c = 128;

            cairo_set_source_surface(cr, rc_font_surface[pair],
                                     (x - c) * RC_FONT_WIDTH, y * RC_FONT_HEIGHT);
            cairo_rectangle(cr, x * RC_FONT_WIDTH, y * RC_FONT_HEIGHT,
                                RC_FONT_WIDTH, RC_FONT_HEIGHT);
            cairo_fill(cr);
        }
    }
    if (!gtk_widget_get_sensitive(widget)) {
        cairo_set_source_rgba(cr, 0, 0, 0, 0.5);
        cairo_paint(cr);
    }

    cairo_destroy (cr);

    return FALSE;
}

/* forward declarations of PNG data below */
static size_t        font_png_length;
static unsigned char font_png[];

struct cairo_read_func_closure {
    size_t         length;
    size_t         offset;
    unsigned char *data;
};

static cairo_status_t
cairo_read_func(void *closure, unsigned char *data, unsigned int length)
{
    struct cairo_read_func_closure *cl = (struct cairo_read_func_closure *)closure;
    size_t l = MIN(length, cl->length - cl->offset);

    memcpy(data, cl->data + cl->offset, l);

    cl->offset += l;

    return CAIRO_STATUS_SUCCESS;
}

GtkWidget *
retro_console_init(int width, int height)  /* returns the console GtkDrawingArea */
{
    struct cairo_read_func_closure cl;
    int i;

    rc_console_width = width;
    rc_console_height = height;

    /* create drawing area */
    rc_drawing_area = gtk_drawing_area_new();
    gtk_widget_set_size_request(rc_drawing_area, RC_FONT_WIDTH * rc_console_width,
                                                 RC_FONT_HEIGHT * rc_console_height);
    g_signal_connect(G_OBJECT(rc_drawing_area), "expose-event", G_CALLBACK(rc_expose), NULL);
    GTK_WIDGET_SET_FLAGS(rc_drawing_area, GTK_CAN_FOCUS);
    gtk_widget_add_events(rc_drawing_area, GDK_BUTTON_PRESS_MASK | GDK_KEY_PRESS_MASK |
                                           GDK_KEY_RELEASE_MASK);
    g_signal_connect(G_OBJECT(rc_drawing_area), "button-press-event", G_CALLBACK(rc_button_press), NULL);
    g_signal_connect(G_OBJECT(rc_drawing_area), "key-press-event", G_CALLBACK(rc_key_press), NULL);
    g_signal_connect(G_OBJECT(rc_drawing_area), "key-release-event", G_CALLBACK(rc_key_press), NULL);
    /* g_signal_connect(G_OBJECT(rc_drawing_area), "focus-in-event",  G_CALLBACK (rc_focus_event), (gpointer)1); */
    /* g_signal_connect(G_OBJECT(rc_drawing_area), "focus-out-event", G_CALLBACK (rc_focus_event), (gpointer)0); */

    /* create font surface */
    cl.length = font_png_length;
    cl.offset = 0;
    cl.data = font_png;
    rc_font_surface[0] = cairo_image_surface_create_from_png_stream(cairo_read_func, &cl);
    for (i = 1; i < RC_MAX_COLOR_PAIRS; i++)
        rc_font_surface[i] = NULL;

    /* create character storage */
    rc_chars = (rc_char_t *)g_malloc(sizeof(rc_char_t) * rc_console_width * rc_console_height);
    rc_color_pair = 0;
    rc_console_erase();

    rc_button_press_callback = NULL;
    rc_key_press_callback = NULL;

    return rc_drawing_area;
}

/*  !"#$%&'()*+,-./0123456789:;<=>?
 * @ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_`abcdefghijklmnopqrstuvwxyz{|}
 * 0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF
 */

#ifndef RC_USE_BIG_FONT

static size_t        font_png_length = 5096;
static unsigned char font_png[5096] =
  "\211PNG\15\12\32\12\0\0\0\15IHDR\0\0\4""6\0\0\0\15\10\6\0\0\0R!\3561"
  "\0\0\16\0iCCPICC\40Profile\0\0x\332\255Wy4\324\357\333\276\77""3c\306"
  "\30\306.Y2D\226lY\263\357d+{\210\302\220mlc\242\"\262\207R\210\222$!"
  "\24I\310\36B\"\312\222-K\221\20*K\264j\336\77\372\376\316\357=\347=\277"
  "\367\375\347\275\376\272\236\353\217\347y\356\373\272\317\271\316\15"
  "\300\312\353\36\34LA\1@@\40\215jc\242Ortr&\341\336\0\36\230\1\3\10""0"
  "\273\223C\203\365\254\254,\340\77b{\24\20\0\200a\31\367\340`J\\\2732"
  "\315\177\374\215I/:Z\1\345\253\264\2\377;\210TG'g\0D\32\0\270\274\377"
  "r]\0\340\362\370\313\355\0\200+\234\26L\3@|\0\200\213\354\343\356\11"
  "\200D\2\2004\325\316\306\0\0)\5\0\242\367_^\17\0D\217\277\3749\0\20\303"
  "\310\3364\0d\34\0\313\21\350\351\33\10\200[\5\300j{z\205\222\1\210\322"
  "\0\340\351\31J\16\0\40^\1\0\331\200\200\40O\0b\33\0H\220\203\2514\0\342"
  "\22\0\3108:9\223\376~\331\245\36@\211\15\0\337\371o-L\10\240\266\24@"
  "\234\357\337\332^:\0\217\7\300\35\223\177k\2336\200\0\0\302\363*\364"
  "\244\242\2\0\0\40\314\372\0\14""3t\372\346>\0\\\6\300N:\235\376\353\16"
  "\235\276S\10\200\236\2h\243\220OQ\303\376\351\27\202\364\3\374_\347\277"
  "5\377\3""4\2\200\2@\3410\216X=\334\10\236\312\264\315\234B\224d\355f"
  "\77\316\361\203+\201\207{W\365n+\2769\1\332\36\234P\216\260\242H\273"
  "\250\231\330\224x\200\304\37\251\24i\21\31\77Yi\271Q\5eEfe5\345\37\252"
  "\352\7\233\324\3254\206\264w\353PtE\365\202\365\177\31\231\30\227\35"
  "\"\230\6\232\215X\250Y\246\35\236\265R\265\276h3c'g\237\3400\354(\344"
  "\344\347\\~l\335\365\300q\277\23""1nY\356w=j\310O=_zM\234|\357\275\352"
  "\263\345\273\343\217\243p\4\10\6\212\7\311\6+\205\250R\17\206\252\321"
  "4Ni\204\251\205\253\234V<s\340\254l\204d\244\3109\276(\266hL\364\367"
  "\363+1S\261/\343Z\342\313\23r\22\23\222\250\27\\\222\15S\366\247\262"
  "\245n]\234\274\324\236Vv9\363JH\272u\206B&W\346\366\325\261\254\306\354"
  "\334k\341\327mrdo0\337X\316\355\272y3\217rK3\237%\177\372veA\342\35\373"
  "\302\275\205kE-\305\251w\235K$J\266K;\312\322\356\35\275/t\177\261\374"
  "Q\305\331\7\272\225\330\312\276\207YU\316\217\366<\232\253.\251\361\255"
  "\225\252]}\\Q\27X/W\277\326\360\2601\270I\276i\275\271\252%\360\311\376"
  "'\37[\357\266y\266\213\266\177xZ\334\341\321)\3249\325\225\373\314\251"
  "\233\273{\350\371\225\36\253^\366\336\241\27i}\26\375,\375\203//\275"
  "2\33\300\17\364\16&\17\231\16\343\207\237\277N\31""9<\3121:>\2263\356"
  "2A\232\230\177S2\3517%7\365u\272y&\376\255\371;\316w\223\263\205s\224"
  "\367\252\363\310|\337\207\234\205\223\213JK\350\245\241\217\267\227\203"
  "WtW9W\27>5~N\373\342\276\246\262\316\264\376n\243~3\375\253\337\226\301"
  "\266\340\366\366\267\201\357e\77\342~\272\376R\373\315\375{s\247\347"
  "O:\335\222N\377\357\376\23p\314\31D\11\326\373l\77""9\316r1p\247\356"
  "\22\342\255\344\323\341o\27""4\332S#\364G\230\"\362U4Ll]\334_\342\235"
  "\324\321\375\2752\7e=\344:\16(+\334V\"*\7\251\14\36\324P\313Q\237\321"
  "\304k\271\350x\350\222\364\\\364\253\15qFv\306wLF\17}7\2236\247Z4\36"
  "F\37\261\262\312\260\36\263\25\260s\263\317w\230v\344q\322sv>\346\343"
  "BsM8\236y\242\300\255\322\375\211G\37y\302s\301k\343\344\37\37F_v\77"
  ">\177\21\212d\200\\\240R\220Z\260V\210\16U'T\235\246zJ1L!\\\356\264\344"
  "\31\322Y\336\10b$\22\271y\356C\324ht\367\371\332\230\374\330\324\270"
  "\263\361^\11\346\211\12I\274I;\27f\223\273R\312RS.\372\\2M\223\274\314"
  "ty\371JozYFB\246\353U\225,\366\254\225\354\366k\31\327\235sDr\226oT\347"
  "F\335<\224\307\2367~+/\337\353\266\314\355\255\202\226;\11\205\26E\354"
  "E#\305\327\357\272\224\10\227,\226>(\13\270'wo\343~]yx\205Z\305\257\7"
  "-\225\347\36j\77\244W\265>\212\2536\254a\250yV{\341\261i\35S\335\213"
  "\372K\15\226\215\304\306\227M\31\315\16-\273[\246\236\334juo\23m[j/{"
  "\352\327!\323\261\336Y\327\25\361L\247\33\325\375\354yf\217k\257x\357"
  "\332\213\346\276\304~\353\227\2/\27_\325\16\304\15\332\14\221\206\326"
  "\206\333_g\214x\214*\214a\306&\306\313&\"\337XO\212O\356L\15M\227\316"
  "\234\177\353\370\356\300,\363\354\307\271\247\357o\315G|pZPY\344Z\334"
  "Xz\375\361\361r\316J\314\352\271O\311\237\343\277\320\326B\327\3436b"
  "7i_i[\11\333\361\337\302\276\207\377H\372y\341W\344\357\250\235\224\77"
  "\247\351Ft:\0*\231\201\204\355a\14b\22%T\263\330\22\377\260\235c\377"
  "\314\31\301\215\345\211\343\345\335}\231\237C\40VpK\310\225\364P\4\275"
  "\367\204h\353>A\361\0\211z)\334~;\351\2132u\262\213\362{\16""8($+6(-"
  "\252\260\253\252\35\264Q;\251a\246\361V\263D\353\211\366\220\316\242"
  "\356o}f\3NC\234\21\2031\253\311\356C\342\246\362fj\346:\26\352\226\6"
  "\207u\217\34\266\262\2646\262\221\266e\265\335\260\33\261\177\354p\375"
  "h\270\243\235\223\2623\247\363\372\261A\227j\327\253\307\303O8\271\251"
  "\270\263\273\177\362\350%\227y&z\271\236T\367\346\366^\367\351\367-\365"
  "\213\367'ST\3\230\3\336\7""6\6e\4\7\206\350R9\251\213\241M\264+\247\310"
  "a\212\341\350\360\241\323w\316P\317\32DpE\314GV\237\213\216\262\214\346"
  "\217^>\337\20\223\34k\33'\30\267\20_\231p&\321$\2113i\372Bi2-E+\25\233"
  ":t1\377\222O\232|\332\267\313\255W\222\322m3\4""3\26""3\253\256\236\311"
  "\322\311f\310\356\273\226u\375x\216h\316\362\215\252\334\310\233\206"
  "y\204\274\341[\267\362\311\267\245n\257\27""4\336\211-4/\342.z[\\r\227"
  "Z\242U\312X:\\v\373\236\337}\245r(\357\257\310{@\251T\177\210{8Ru\367"
  "\321\251j\375\32\266\232\267\265\225\217\343\352\34\352%\352w\32&\32"
  "\2375=h\276\336\222\360\344T\253W\233}\273\361S\265\16\351N\341.\336"
  "g\354\335\204\347\330\36T\17\275\367\327\213\237}\277\372w^\241\7\10"
  "\203\234C\2\303\342\257\25G\364G\355\306\274\307#'2\337\334\233\354\234"
  "Z\230A\275\25~\247\77\3535w\361}\355\374\354\2nQw)\361c\333\362\332\252"
  "\334\247\250\317m_~\254kn\\\332\34\334\"l[\177+\372\276\374S\352\27\355"
  "w\337\37V:\77\235\16\3607\373\0\0\260*\0Y\274\0v\315\0\266\306\0I^\0"
  "\342\267\0xJ\1\254X\0\354\324\0\25\252\10\250\343\365\200\314\232\374"
  "+\77\0\1""4`\1\17D\340\2A\20\7\5\320\201\303\340\12\201\20\15\351P\14"
  "\2150\0\213\10\40|\2102b\205P\220d\244\24y\216|D\21P\262(\33T8\352&\252"
  "\3\265\212\346A\353\242\375\321\327\320\235\350\15\214\10\306\32\23\203"
  "\251a\330\303`\313\20\315P\300\320\201\335\217=\214\245bWq\3228\177\\"
  "9\3563\243\2\243\17c\6\343\23<\26\177\3\277\304\244\314\24\316T\314\364"
  "\232\40N\210\40\14""0\357c\216d\36f\221fIaY&\232\21\357\263\262\260\206"
  "\260\216\261i\263\25\261\23\331\317\262/r8p\364pjrVqIr\25q\313s\267\362"
  "X\363,\354\212\342\335\305[\263\333f\367W\276k\374\232\374\357\5R\4\25"
  "\5g\366\244\12i\12}!\225\10\273\213\10\212\274\331{C\364\204\230\230"
  "\330\247}\217\305c%l$\367I\376\226\32\335_-\235)\23&{L\316P^\376\0I\201"
  "S\21\253\4J\177\224\351\252\230\203\4""5Nu\1\15QMM-;m\212N\242n\241^"
  "\273\376{C\26#Mc\77\223\353\207^\230\341\315\355,n[~9\242c\225l\375\336"
  "V\317\256\320\201\341\250\217\343+g\365c\345\256b\307\357\270\11\272"
  "\337\"\213{V\237\324\360\36\360\365\361\7\312\235@\223\240/!\271\241"
  "f\264\235\260\332\323\1g%\"\276\234\253\213\216\213\261\214\343\211\237"
  "Kl\272\220\235\22x\3212M\361\12w\372V\346HV\327\265\252\234\314\334K"
  "y\361\371a\5\201\205\336\305'KN\224\35\275oQaUi]eS\355\\\353^\347\323"
  "\340\337\24\334\22\336\32\333~\251#\273+\277\273\274\247\346EK\177\327"
  "\253\301\301\231\341\225\221\255\261\2357\330)\316\31\322;\3719\255y"
  "\243\5\253\245c\313\336\253a\237\23\327nlT|m\333\36\372>\367\213i\307"
  "\223N\377\17\376[\202+\4@4d@)t\300$l\",\210\4b\210\270!Q\310M\244\31"
  "\231A!(1\2249\352\24\252\30""5\200\372\211\26E[\241\243\320\25\350i\14"
  "+F\37\23\206\271\217\231c\340c\40""3\24""0\314`\371\260N\330\353\330"
  "\11\234\40\316\3W\212\373\304\250\314\30\3138\212W\300\307\343k\360s"
  "L\341L}\4a\302i\302\40\263\14\363\5\346e\26{\226gD\31b\6\361;\353\11"
  "\326n69\266\233\354x\366p\366E\16G\216>N=\316\6.%\256:n\3\356\21\36\177"
  "\36\372\256k\274\362\274/v\373\360a\371J\370M\371\77\13\\\25\324\22\374"
  "\270'[\310X\350;\251\\\230,\"\40""2\2667K\324^\214Ol~\337\3\361\263\22"
  "\306\222<\222+R\35\373\363\244#d\\e\365\345\244\3449\17\300\201-\205"
  "e\305\31\245q\3457*S\252s\7W\325~h\3404\371\265\224\265\17\353\370\351"
  "^\320+\323\3577\370f$e\354b\222v\350\271\31\233\271\217\305\323\303{"
  "\216\234\265\32\267\321\266-\267\337\345\220pt\333\311\307y\326\305\315"
  "\365\303\11\177\2675\2170O\242W\201\267\242\317\240_\20\2055\340q\320"
  "\361\20&j5\215\34\266+\274\357LB\204^\344NT\333\371\304X\343x|\302`\322"
  "\255d\377T\265K\254iKWZ32\256R\262-\256\313\344\374\310\35\312\253\310"
  "O(\40\27\352\25\213\334\245\227N\337k.\317{\20\361\320\365\221~\215\304"
  "c\\\335f\303xSSKAkJ{h\207s\227^\267D\17S\357v\337\314\313\256\201\7C"
  "7^'\214R\306\255\337\250O\261M\327\274\265\231\3357\367s>wA}\211\370"
  "q~%\345\223\354\27\374\332\207\215\313_\325\266y\276}\375Q\364\313m\347"
  "\374\77\376\243\0\3""8`\2""6\340\5a\330\17*`\0""6\340\16!\20\13\331P"
  "\16\235\377L\200$b\204\270#1H\1\362\24Y@\21P\7PN\250\4T5j\16\315\216"
  "\326CS\321\305\350I\14'\306\22\223\204\351`@3\2302\2443\214cIX_l-\16"
  "\215\263\305\25\343~2Z3V\340y\360\226\370h&u\246\333\4<\341\24a\226\331"
  "\212\271\203E\223\345)Q\227\330\314\252\301\332\300\246\311\326\316n"
  "\312>\301A\346\330\346\214\341\342\341z\304\355\302C\340i\336\25\302"
  "+\301;\277\273\210\217\314/\301\377Y\240F\360\334\36\23!.\241%R\233\360"
  "\15\21\332^kQU1\276}\310\276\217\342#\22\235\222UR\25\373K\245\213e\212"
  "e\357\310\225\313\327\37x\2460\242\270\244\214V\21V\325;\350\251\226"
  "\244\376@cB\213\240\255\255C\325-\323{g\260\327\320\317\250\306\4""9"
  "t\310\364\232\331\262\205\226e\316\341M+\13\353\7\266<v1\366\237\217"
  ":8v:+\37\253p\225:~\317M\302\275\214\254\340\331rR\317\373\225\257\233"
  "\337'JR\40)\250!\304\201\272N\273\34\246\20>|&<B\40\262=*\350\274@\314"
  "\363\270\340\4\301\304W\27\342R\224R\227/\335\275\354\221.\2301t\365"
  "b\366\241k\77r.\344\222n6\3342\317\237*\10*D\25e\335\225(i*3\2777Y\356"
  "W\361\2752\261\212\373\321\335\32\345\332\366\272#\365\343\215'\232\26"
  "Z\2\237l\267\235{\212\352\210\357\302<Kx\216\351\211}\201\364E\365\257"
  "\275\12\30X\32r\37\236\30""92\372t\\i\242pr\317T\322\364\364[\245w\371"
  "s\354\357\303\347\273\27\370\27c\226\276,\333\257\224\254\356|\366\372"
  "2\266\256\275\221\265\271\262e\277\335\371]\346G\342\317\211\337\306"
  ";\337\350t\200\277\373\22\0\0""0\31\4Q\202\250$\13\3C\370\377E\0\345"
  "\324\277\336@\0\200\331+\320\336\26\0""8\0@\0\14\40\10(\20\4T\40\201"
  "\5\30\200!\300\337]\15\0\0\313\6p\223\3\0\240\336\222\372\77\356\245"
  "y\235\246\1\0\30\4\5\237\241\372z\373\320Hz\301\301\24/\222AP@\360)\232"
  "\27U\232d\32H\226\225&)\310\313+\3\0\374\27\214\210\372\4\27\307\245"
  "\324\0\0\0\6bKGD\0\377\0\377\0\377\240\275\247\223\0\0\0\11pHYs\0\0\13"
  "\23\0\0\13\23\1\0\232\234\30\0\0\0\7tIME\7\333\2\3\25(\22hQ\4\273\0\0"
  "\5iIDATx\332\355]\321\222\344\40\10\334\271\332\377\377\345\354\313M"
  "U*%\320\40\20M\272_nn\35\25\25QZe~~\10\202\40\10\202\40\10\202\40\10"
  "\2020q\34\307\321YfE}\335\375R\321\276k\276\177TM\202\330\337\260F\14"
  "B\225Q\266\14\3279\335\372\376\233\26""4\202\40\210n;\346\261\301\336"
  "\357g\255\33\22568R\257\267\317f\277sMG\326X\256Q{\315AD'8\256\4a\343"
  "\227]@\20\3738\322\237\317\347\343\311\363\375\376\371\363\271<\351\357"
  "Z=\21YFu\235\313\30\245i\371V&:F\262Jm\320\362Yej\3437\352c\255\314h"
  ">T\316\225\306'\322\206\221\316[eJ\345\242}\211\214\23\322\216\221\216"
  "\214d\321\344D\346\253fo\264y\360\375\233&g\3268D\373\362\233\2569\25"
  "V\33\320\366x\372%:\266\232~i\377\327\312\336\311^\317\330\221\325\333"
  "\206\214\21\307a\317v\40\373\245\212\375\13A\20\4A\244/\200\32\253o1"
  "\376\243\317\322\277\221\317\232\214\232\323\246\245i\362!ev9\311\310"
  "\377\21\331\255\276Et\302\323\307\350wg\345\314\356\357\214\357V\364"
  "5\332O3\375\27\31""3\264}\221\271\351m\207vck\224\226\321\276H\276\350"
  ")\273\267L\255\315\222\335\213\352O\3668f\314W\317\372Qe\373\253\353"
  "\274\313&>\235\330xz;\272\236$t\254\341\324\303z\235x\263^\363)\12A<"
  "\10_v]:\11\274:\16Z\31\222\301@\31\374\317\177D\312\31\235\206\236O+"
  "w=\241\321\344\266\372#Z\256Ww\252\333\217:\271\31\355\264N\325\265\362"
  "5\2\352\253\333\36\342dt\322=+\357l}\335:^U\7\332o\321|\253\331x\351"
  "\206]\366\30\315\326u\266\327\221vJ7W\242vD\312\253\225\251\345A\313"
  "\364\220\243\210\34""3\204\347H\376Q>\211`\364\366\213\247\35Z^)\237"
  "6\16\21\22\320\"\357\242\304\3421\300\252\373\223\263\214;\332H\2028"
  "\343w4!=\327t\231\3064\246\315\247Um\206GW\27\245\205\353\374w\211L\270"
  "~\247k\321]\205\334\210^\35\355\224aV\317V!\221\220g6\2436Ft\305\272"
  "\302O\344\351\312\310\316\314\214\335\316\204M\246\235\177\342\23\20"
  "\353i\14\372,\7\265+Z\31Z\37[\266J[\213%\233\203\330>\257\236^\35Wm\237"
  "\340\351[O>D~\353@\240J\3273\313E\237\31\316\254U\210\216\241\351\35"
  "{:\202h%6\10\202x\17\316,\275\364\206\32y\253^\271\231\326\342qD\34\371"
  "\310\215\225\273\235(o\177[\33\304H,\220\221\363)\245y\372\326{S(rs\303"
  "\323\366\350\270u\227\31\251\363\356'B+;\316\232^g\222\270\222\23y\275"
  "!\241\315\301j\242r\364\374\317\3534g:\237V\376]\3422U\3112C\264g:\266"
  "3\355{\3y\350\215\257\203\310#\331\254\363\355B\204l\352\40\214\236\4"
  "\317S\12\315>Y\373\266\312=Gt\317V\261\77\210\356\347Il\20\304C\310\10"
  "\257Q\321N\2434\7p\264x\"\213c'\301\341\225c\345\300\226\326s\25\355"
  "FA\264\17\320|h@\301Lg\327\"&\254q\215\236H!\327\225G'\256\231s\"\352"
  "\314i\355\257\32\243\12[\20\271\12\35\335\14\241\201R\253\11\\kNGI\217"
  "U\202\23\356D\240=\341\35\373\23\366:\331\244\261EPD\346\30BzXkQ\326"
  "\215\15\211\334\320\352\272\312Mrcn}\366\254\263\221'\312]{\214\312~"
  "B\311\237\31\374z\7\213iLcZ~Z\266!\322\30{d\321\264\362\276q\1\254h\357"
  "\354\323\226L\231V<\345\224H\204H;\220\30/\332u\36178\24Y$T\367\306*"
  "\343f\200u\222\331\265\21\274\363\215\373(~@\2248\316\30c\224\334\\I"
  "\227\272\306nE\22\252j\34:\210E\315\271\217\336\330@\236\215\316\356"
  "\355\244\362F\237Gy+H$\202\270\23\14\36J\20\2338\325\325Q\340\273O%;"
  "H\203\310\311o\327\306\324\372\211\333\2257\325h}\263\301Cg\202\201\""
  "\262j\327z+\365=[\357\357\16\370\226]\277\365\34*\22\14s\5'Pz\2\210\352"
  "\310L\0[T\276s]\273:9\236\371\227aO#\343\272\352\334\273k\255\350\250"
  "o\226\234\260\202\234w\355\247,95\362Bj\303\316\363\235\40\316\340S\24"
  "\202x\31\244\310\367\332B8Z\20W\270\265\321U7r\3753\"\223t-/Z\246'\36"
  "\306\235eV\215q\244\376\254\361\363\304&\321N\303\243W5g\237ph\1=g\35"
  "\213l\362&\372K9\321\230$\31se\5gr7\307\305Zs\"\363\17\11^\333e{\262"
  "\373\251\353`\2\15\214n9\324Q\275\215\214\355\323\347\237'~\12\11\14"
  "\202\40\10\342q\270c\321\37\375<[U`\303\312\337z'\326\32\37\202\240\316"
  "\305\373\304\343pz\373\361\20\300\265\365\31\353\217W\27""8\326\230]"
  "B\1772\230\330O/v\271\221\273R\333\271\77'\10\202\40\10\202\240c@4\352"
  "V\346\223\321\35\364\365\315\304\6\355\11\321\351\334\223\330\300\312"
  "\374\3""6\224\311\370\202\17j\370\0\0\0\0IEND\256B`\202";

#else /* RC_USE_BIG_FONT */

static size_t        font_png_length = 4565;
static unsigned char font_png[4565] =
  "\211PNG\15\12\32\12\0\0\0\15IHDR\0\0\5j\0\0\0\22\1\3\0\0\0\370\311u:"
  "\0\0\15\357iCCPICC\40Profile\0\0x\332\255\227y8\224\357\367\307\357Y"
  "\314\30\313\330%\353\20Y\262\257\331\267\2205{\21\205!\3736\6\205\310"
  "\276E!J\262\357eI\310\236=\21e\311\226\245H\10e\213H\362{\324\347{}\177"
  "\327\365\333\376\371\235\353z\256y=g\356\347>\367\334\357\363\314\271"
  "\17\0TL\266^^np\0\200\273\7\221`\254\245\206\273da\211C\277\7\30@\1\220"
  "\0\6(l\361>^\252\206\206z\340\177\264\3351h\34d#B\307s\205\267K\21]'"
  "\336k\365!B\304\341\316\222\253\340\1777,\1\12\10\0L\20bz\307\277\254"
  "r\314v\177\331\364\230\375\211^D\210\235\216\31\357dk\17q\20\304\202"
  "\4Scu\210K\216\347q\374\313\365\307l\367\227_\35\263\37\336\361\370\331"
  "\11\0P\264\36\366\316\36\0\240\327\40V\262w\360\301C_\37\307\265\267"
  "\367\301\273C|\27bawwOh~l\33\304|x/\2\364,v\31b\241\343}\371\273d+(\206"
  "$5\0\230\256\177\373\3748\0\250\205\326\301\313\374o\337\251#\0\30\355"
  "\0\310\323\372\267o\333\370\317^\301\30\337\372\\\223\20\377\343\202"
  "Q\250\1@2{t\264}\32Z[2\0\207IGG\7yGG\207\371\0\40\246\1hs\303\373\22"
  "\374\376\331/\30l\0\200\377\353\376\357o\376\307\20P@H`8\32y\11\245\212"
  "\36\305\20\310v)\342\260\374T=4Wh\367\351#\31\31NT\2374d\236g%\262\243"
  "9\3229%\270\332\271ux\246y\335\371~\13\304\11r\11\271\10\13\212\214\211"
  "KIPH\311J\355\313\310\235m\222\223\225\37V:\251\354\246\302\255\352\245"
  "v\240\241\245Yz\236\\\333CgTOV\77\321`\316P\306\350\266\361\254\251\210"
  "Y\244\371\310%\16\13\27\313\262\313\233\326bW\\\256\206\332\244\332\26"
  "\331\325\340;\354\3378L^\373\344\270\346\264\343|\350\212v\243ug\363"
  "\340\365\24\366\222\364\226!\234\365\221%\312\373\312\373\311\372K_\227"
  "\270!\26\40\34\310\37\304u\2239\230:\4\31\262wk5t:\354MxKDYdzTd4!\306"
  "*\366\\\334\231x\352\370\235\333S\11\355\211\245wR\356z'\31%\213\247"
  "\320\247\354\336\33OmL\313\270\357\377\3008]\370!\305\303\225\214\356"
  "G\2172\335\262\24\262)\263gr*s\243\362\314\362O\345o\24\264\24\306\27"
  "Y\26\363\25\357\226t\226&>\276\370\204\343\311R\331\263\362\200\12\225"
  "JTe\377\323\324*\313g\354\317\346\253\213k\234k\5j\327\236\227\327y\324"
  "\213\324o4<m\364j\22m\332l\256j\361xq\346\305\227\326\2426\373v\356\366"
  "\317\35\205\235v]\34]\323\335\31/-z\30z\206_\335\3555\354\243\351\33"
  "~\235\330\2577@90\364&\341\255\316\40f\260o(vX{\4""3\362\352]\334\250"
  "\301\30\355\330\304x\372\204\325$nr\341}\361\224\313\264\310\364\367"
  "\231\346\331\210\17\272\37\351>N\315\345\317\273}\222Y\200-\364\177N"
  "_\274\266$\271\214X\36\376\222\263\342\265\252\262F\267\266\370\265\361"
  "[\342\272\355\206\364&\331\346\307\255\372\355\244\357.;\352\273l\273"
  "\273\77\6\367J\367\303\177Z\37\310\376b\370\265}\330\373;\351H\377\350"
  "\350\77\353O\216\246H\306\362Q=\241\376I\33@O\302\20\177\202\203\251"
  "\222Y\231\245\235M\203\275\206\3437\247\33\327wn\77\236M^W\276\217\2"
  "\27\317\364\11\235\25\266\23\351\24\223\22\317\221\304JyJ\17\235\225"
  "\227M\227\233U\300(Z)\333\251\340T\255\324\252\317\2415L5\363\264\306"
  "\316\357\351\10\352\22\364\32\15\20\27\14\15\223\215\306MXMm\314\262"
  "\315g.1Z\250ZZ^v\262\"ZG^I\271\232kSi\373\302\256\37\77i\277\350\260"
  "u\355\267\23\2513\215\13\263+\227\33\277\273\210\207\244\247\254\227"
  "\242\2672A\331G\216(\343+\341'\356/r\235\377\6.\200)\20\33\4\13\332\276"
  "\3719x,\244\347VmhvX|x@\204C\244n\224x4S\364a\314\\lw\\i|\334m\247\4"
  "\355D\376;dwV\356\366%\225&G\246X\337\223N\245I]Mk\277\237\374\3002\235"
  "+}\345auF\360\243\363\2314\231\23Y\231\331\16""9B9;\271-y\221\371z\5"
  "4\5\243\205\17\212\254\2129\213\227J*J\335\37\213<\336zRW\346_.[~P\321"
  "Ry\363\251\322\323\243\252\326g\341\325\347jHj^\326\306<\327\256#\253"
  "{]\237\320\240\337\210m|\323\224\334l\336r\262e\372EV\253m\33w\333r{"
  "i\207K\247P\347fW]w\340K\345\36x\317\313W)\275\326}\274}\33\257\233\373"
  "\243\6\214\336\260\276Yz[;\30>d<\214\33\336\30i\177\227<j7&>\216\34\237"
  "\234(\235\14zo4\305;u8=<S2{\353\303\245\217bs\24s_\346;>e-\4~\266X\224"
  "^\242_\332Z~\367\345\371J\372j\350\332\315\257\261\337\"\326\211\33>"
  "\233\341[a\333\304\357\304\235\310\335\210\37~{\376\373\321\77c\16\202"
  "~\5\37\306\375\276~\244\361G\377X\22\34\252\227\324\223\214\233\274\232"
  "\322\4\373\233\372&\3157\272@\6\24c8\23\323\311;,\264\254al;\34\326\270"
  "\247\\\210SW\271[O\263A\377\0\365\2\3503\246\202\267\205\352\204\227"
  "D\331\305\314\305c%\32$\227\244idd\317\32\313^\223\327\221\377\240P\254"
  "\370BiXyI\345\227\32\205:\35\224\7$\232TZ'\317\363j\213\352\310\352*"
  "\353\311\351\253\33\250\\00\3247\3220\26""4\2412\3312\35""5{n\376\340"
  "\242\377%S\13)K:\313\315\313CV\325\326\367\256\370_\265\260\221\266\245"
  "\261\375j\327\207/\265\217r\260\276&\347\310\340\270\3514\340\\\342\22"
  "\341\212w\223q\247p\377\344\321\350\231\354\345\341\255B\240#,\3714\21"
  "\357\372\342\375$\374\21\376\303\327\363n\20\2\324\3\351\3\27\202\252"
  "o\206\4\353\207\260\204\254\334j\10\215\15""3\11g\13_\214\250\214\274"
  "\21\245\25M\27=\23S\22K\214S\214G\305\17\337\316NpJ\24M\374q\247\365"
  "nt\222I2[\362RJ\325\275\33\251\312i$i\375\367S\37\\I\347\206\362\244"
  "*#\350\321\271L\362\314\221\254\254l|\216@\316fnc^X\276n\1C\301\207\302"
  "\342\"B\261b\11i\311Hi\316c\227'\222e\240l\240<\263\302\255R\356)\372"
  "\351hU\3213\337j\265\32\352\232\17\265\225\317\303\353\314\353\371\352"
  "\17\33&\33_6U4\77h\211|\341\333\352\320f\326\256\331!\333)\330\305\331"
  "\315\364\222\246\207\374\25\252\27\336{\324w\360\372g\377\301\300\341"
  "[\304\40\371\20\3350\353\10\357;\211Q\2651\323q\307\211\240\311\224\367"
  "\217\247\272\246\27g\341\37""8\77\252\3159\314\337\376T\2730\267\210"
  "^RY\216\372\322\266\262\261&\3625\370[\333\372\376\246\302V\302\366\320"
  "\16\371\256\321\217\202\275\225\237\2\7\304_\375\277\251\216X\216\365"
  "\377[\373\216\15%\15@*\23\0\246\315\0\230h\2\20\355\0\325\255,\250VA"
  "\365\313\220\22\362\313\2\270\217\4\200_\251\7\2609\255\177\325\17\250"
  "n!\0\12:\27`\1=`\3\274@\34(\3\3`\15<@\10H\2\205\240\21\14\202%\30\200"
  "1\303\244`\20607X,\254\4\366\12\366\5N\16\27\206\33\303\375\341\217\340"
  "\235\3605\4#B\5\341\212\270\217\350Bl!\271\220F\310Pd\15\11;\211\11I"
  "\10I.I'\352\14\312\0E@\255\241\5\321\256\3502\3647RqR'\322d\322\27\30"
  "\24\346!f\231L\212\314\237\254\220\354\35""9/y\40\371\40\305i\212\40"
  "\212\21JA\3128\312\25\254\16\366\11\25%\2257\3258\265\22u\1\15\226&\200"
  "f\211\326\234\266\227N\201\256\212\236\237\276\200A\224\241\225\321\210"
  "q\361D0\323\11\246\232\223\306'\2773\337gQ`\371\304\32\307&\3016\313"
  "\36\317\241\300\261\216+\346\264\345b\343z\177\352!\367U\36\36\236\257"
  "\247\237\363\206\361\31\363\237\346\377%0v\246Z0E\310O\370\262\3109Q"
  "Q1\2348\235\4J\22H\376\226:\222A\236%\227\245\223c\225\347VPP4UrS\216"
  "R\311WmW\373t\216RCA\323E\353\301\371\327:\30]S\275\34\375\365\13\312"
  "\206\261F\237LTM\363\315I.:]zk)w\271\314\232\347J\236\15\233m\26\236"
  "\327\276\372\232\274\343\240\263\223+p\313\363\320\362\\\367\316\360"
  "\321!\36\372\325^w\17\340\13\\\277Y\27\22\36\252\37\316\30""1\37\325"
  "\24\223\26\347q[\77Q\342.C\322N\312hj\367\375\252\364\224\214\204\314"
  "\210l\277\\\217|\307\302k\305WK/>\321+7\2544\2522\256\266\254\265\255"
  "sjpm\362j\361o\15kO\350L\353\316\356)\353\255y\3352\320\375vhhvdutg\374"
  "\360=j\232n\26\367Qt^qAc\321p\371\362\212\343\232\337\267\250\215\207"
  "[\345\337\333v\207\367\346\17\310\16\355\217\363\347\277\325_\37\322"
  "\337\35\322\77\31\224\200N0\5\266a\2240>\3309\230\15,\30\366\10\326\14"
  "\233\205\303\340<p]\270/\274\20>\10\377\211\340F\30\"\202\21\345\210"
  "\31$\25R\15\351\207|\202\234'a&\301C\332\317\242\230Q\26\250\7\250I4"
  "\33\332\16]\202\376J*E\32F:\206\21\307D`j0\363\220\362\375\344\234\344"
  "\327\311\207(\204(b(V(\315(_b\205\260\311\330=\252\253T=\324\"\324\217"
  "h04\376\220\352\227h\373\351T\351\32\350%\351\353\30\324\31F\31]\31\217"
  "N\334g\22ez}\322\211\31\305\\\314\242\315\362\215\365\36\233\"\333\27"
  "\3664\16M\216=\\\31'\236\213\225k\374T*\267\31\17""3\317\302\351\12\336"
  "\0>M~F\376U\201\3163\231\202\201B\326\302j\"\2\242tb@lG|EbVrB\352\275"
  "\364\264\314\374\3315\331}y\264\2\213\242\224\222\201\262\213J\214j\251"
  "\332\200\372\17\15\1M+\255\304\363\257t\250u\235\364:\14\330/\4\30N\30"
  "+\231\224\231\2350\217\274\270k\341d9gec\375\371\252\253\315\206\235"
  "\237=\326!\327Q\302i\310\305\323\215\312\375\271\347\25o2B5\21\357w\302"
  "\277\377Fd\240j\320ap\333\255\2500\315\10L\344PtV\254k\274l\2U\342\362"
  "\335\326\344\344{niz\17\204\322\3673\2063\313\263#s\361\371\252\205\\"
  "EG%3\217\233\3132+\2\237Z\77S\253\341{\216\256\333n\230hjj\311m\215k"
  "\367\351\264\354V\355\341\353%\353\333\355\237}\323=X1\374\360]\344\230"
  "\333\204\321{\271i\352\231\232\17\306s\247\347\177.d,\312-c\277,\254"
  "\306}\25^\307l|\336\272\363]v\227\361\307\367\375\202\3\233\303[\377"
  "\350\17\207:\2""4\40\3\324\200\11p\2023@\32\250\3c`\13\274A\30H\3e\240"
  "\353\237\14\340\207i\300la\241\260\\X\7l\21z\377\305\340\26\360Hx5|\36"
  "A\203PE\20\20\205\210)$\35R\37\31\215\354$A\220h\223$\221L\240p(gT-\32"
  "\2016A\27\242\177\222\32\221\226c\30""1\372\230\20""29\262\34r\14\271"
  "/\371\34\205!E'\245\2e\7V\5\333L%O\325@\255@\335N\243M3I\213\247\335"
  "\245\13\245g\244\177\306`\305H\316\330|\302\233\211\217i\341d\1""3\236"
  "\205\17\322\272\206\355&\273\26\7=\3072\256\215\363!\27\361\224\21\267"
  "\14\17\363i\330\351/\274\243|]\374U\2\345gJ\4\13\205\12\205\363D\312"
  "D\353\305^\212\217J,K!\2449eT\317\332\313F\313U\310O*\222+))\23TJU\77"
  "\252\237:\347\242Q\243\5;\177^\373\276\316\212\236\242~\272\301\266\241"
  "\236Q\205\11\243i\250\331\267\213\346\227\272,\245.\227[\13\\yl\303g"
  "[\212\27\267o\271\246\352\370\326\331\306\345\253[\264\7\316\263\301"
  "\333\234\260I\274\3\235iFn\370\7\262\6\265\7{\336b\15}\25\356\25\311"
  "\26\3656&<N2~%\241\350\216]\22[\362\360\275\333i\347\357\357\247\307"
  "d\340\36""5d\351fO\347z\346\303\13R\213\370\212\233Ju\37O\225\271\224"
  "\357UFU1<+\252\221\252m\257\273P\77\321x\265i\21:\215\356\266\335\354"
  "\200wFt#_F\276B\366\206\275\206\365\7\17l\274u\37\\\36\266\35\231\34"
  "\2750\3261!9\231\77\305>\35=3\363A\362c\366<\315'\377\205\236E\226\245"
  "\320\345\365\25\263\325\342\265\303o\16\353\343\233J[\251\333\253;f\273"
  "]{B\373Q\77'\177i\36\3768\326\377o\277\364\307\310\324=\335<\11""8=\365"
  "s\340\377\327\334\335|\377\25\343\270K\242p\36003\201>i\241\213\25\312"
  "5O\340\6]\4\200\3z\320\35\24\374o\257\366\247\246A\375\330\243\343q\240"
  "^\237\360_\346%:\\\377\323\17\251{z\335\40""8;:\21q\252P\247\352\200"
  "S\367t\367\362%:\20\4q\332\36xaA\234\270\250\250\324\361\270\377\0\214"
  "\210\372\4/\327\242\275\0\0\0\6PLTE\13\0\0\377\377\377\317\36o\36\0\0"
  "\0\1tRNS\0@\346\330f\0\0\3\202IDATH\307\325\326M\213\333F\30\7\360gF"
  "\303zd\266\316\3101t\3f5\326\12\222\334\226\236\366\20\310\310\30\342"
  "=\244\335\313\336JI\367\23\344\270\24\303\216e\321Ui\10\362\346\320\336"
  "\\\12=$\267\34s\10(/\260\271\346\33\310K\16""9.=\271P\342<#\331&\315"
  "n\223m*B\242\223\306\32\215~\32\377\347\31\1|j\307\336tq\214\261)\305"
  "t\232\272\323\211\222\2\330\307\225\210\317\213\313\77\22\2271\235\317"
  "\16\343\37\10\265R\222\301\373\36\227\15tY\334H\367y*8\243y\2336Y\345"
  "\\\2666\331}\325\254\25=6\232\314\265\16\237\374\365\367F\340V\236\371"
  "\324f\275^\315\255\34\376i\2630\220\243\307LHY\304A\300\6\250e\22[\372"
  "\255\207\20\320\12\340J9\334\301\321P\"\267\256\362\266\267\314\265\334"
  "\1M=\307\14\11\272\343UE\344{\4:\1#\274\203\334\260\355\210X\2069W\267"
  "\221\2735\347v@\231`\350\5s\301\15d\275]\16""7:@\256\240\325~\336\16"
  "\32B\251\337\11\236H|\252\2\25^\257\212.\253\23\10\15""7\262.D\250L\272"
  "`\270\370J!\2252~\77W%\376\2032\270\340\276\350\267$\10Z\311\7\347\201"
  "\17\252\217s\311<9\40Z\203\244\333U\21\33\256F\256\315\350\335\316\214"
  "\313\343\260Q%\224\250\26>\314\375r\377\311\363c\365\303wM+V\203\373"
  "\273\275okw\366\17\217\314\77\40F\217\220\333\355\376\361n\356\362\222"
  "Etzc}\345\35'\330\217\205Z\2450\237]\21""0\350\207k\343\233\274\351\14"
  "H\337\232\10\266\355\270\234\235\37\37\353=F\252\214n\372a[&\373)r\17"
  "0\361\324\312Lt\341\327\256\377\25W\321\367\14b\35nF\341\216s\253\333"
  "2\311\361\204\376\21\364\272\237<,\207\333O\325S\344\326\277\311\254"
  "\211\341bJ\245\23s\257\246\1\227\37\347\333n1\273\267\361\322\27\214"
  "\332\215\260'G\40\13.\256P\235s\23\14\214}\34\341\333\376\4\341\246\211"
  "u\334%\24\3734lb\302\340'w\337\340N\317\351\336\325\342l\2627\236N\243"
  "3W\"7\373mx\11\2714\17\203\14\20p\361\330f\"\300\10\342OBl;of\227\332"
  "\313\241\222\7\206\273u\320`|6\273.r\317O\224\341\376<\347V\36\31.\267"
  ")9\21\206\314\324\212\305\354\2\77\3736\301\364\360\262@n\2610T\300\30"
  "H9\343\302`\"\305\3272\231s\235\2422(oXp=&f\263\233\367\1\265\213\334"
  "[z\306\205\26""5\257d\263b\251\265K\340b\30\242\341\345K\346\6\2)\254"
  "+\217q\256<\21\13\217``\2602\40>\231\27""2\207\3739\327\24\262\276\330"
  "\272\27,\270\255\"\273\270\34!V\206\273\203yj\325W\266\356yK\274(dW\312"
  "\341\16\216\304\323\342\6\344\352\346\325\321\241ne7Es\25\271Xw\305\252"
  "\34\231m\2p\233\250\361\213\270;\364T-\271\226\352L\342\13\270\217\13"
  "\356\352/ye\350\5.\304kX\20\2602\304\225t\374\22\373,\271\351\211]\355"
  "\303\303\0y\231\3147au\332\225S\17\365\217M\230\274\232\267B\\\233z\326"
  "C/fd6PI\334\177\325\221\377<\302i\\\377\224O\234R\270\377\377\350\1T"
  "\340-\356\312\247\313=\353\7\344I\356ke\27""6sBx\323&\0\0\0\0IEND\256"
  "B`\202";

#endif /* RC_USE_BIG_FONT */

