/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QDebug>
#include <QScriptEngine>


/////////////////////// Local includes
#include <libmass/MassSpectrumJsPrototype.hpp>
#include <libmass/MassSpectrum.hpp>
#include <libmass/DataPointJs.hpp>

#include <globals/globals.hpp>

namespace msXpSlibmass
{


	MassSpectrumJsPrototype::MassSpectrumJsPrototype(QObject *parent)
		: QObject(parent)
	{
	}


	MassSpectrumJsPrototype::~MassSpectrumJsPrototype()
	{
	}


	MassSpectrum *
		MassSpectrumJsPrototype::thisMassSpectrum() const
		{
			MassSpectrum *massSpectrum = qscriptvalue_cast<MassSpectrum*>(thisObject().data());

			if(massSpectrum == Q_NULLPTR)
				qFatal("Fatal error at %s@%d -- %s. "
						"thisMassSpectrum() would return Q_NULLPTR."
						"Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			return  massSpectrum;
		}


	int 
		MassSpectrumJsPrototype::initializeWithTrace()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{

				// The single argument must be a Trace

				QScriptValue arg = ctx->argument(0);

				QVariant variant = arg.data().toVariant();

				if(!variant.isValid())
				{
					qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
						<< "Variant is not valid";

					ctx->throwError("Syntax error: function takes one Trace object.");

					return -1;
				}

				if(variant.userType() == QMetaType::type("msXpSlibmass::Trace"))
				{
					Trace trace(qscriptvalue_cast<Trace>(arg));

					return thisMassSpectrum()->combine(trace);
				}
			}

			return -1;
		}


	int 
		MassSpectrumJsPrototype::initializeWithString(const QString &xyFormatString)
		{
			return thisMassSpectrum()->initialize(xyFormatString);
		}


	int 
		MassSpectrumJsPrototype::initializeWithLists(const QList<double> &mzList, const QList<double> &iList)
		{
			return thisMassSpectrum()->Trace::initialize(mzList, iList);
		}


	int 
		MassSpectrumJsPrototype::initializeWithRangedLists(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd)
		{
			return thisMassSpectrum()->Trace::initialize(mzList, iList, mzStart, mzEnd);
		}

	int
		MassSpectrumJsPrototype::initializeWithListsAndRt(const QList<double> &mzList, const QList<double> &iList,
				double rt)
		{
			return thisMassSpectrum()->initializeRt(mzList, iList, rt);
		}

	int
		MassSpectrumJsPrototype::initializeWithListsAndDt(const QList<double> &mzList, const QList<double> &iList,
				double dt)
		{
			return thisMassSpectrum()->initializeDt(mzList, iList, dt);
		}


	int 
		MassSpectrumJsPrototype::initializeWithRangedListsAndRt(const QList<double> &mzList, const QList<double> &iList,
				double rt, double mzStart, double mzEnd)
		{
			return thisMassSpectrum()->initializeRt(mzList, iList, rt, mzStart, mzEnd);
		}


	int 
		MassSpectrumJsPrototype::initializeWithRangedListsAndDt(const QList<double> &mzList, const QList<double> &iList,
				double dt, double mzStart, double mzEnd)
		{
			return thisMassSpectrum()->initializeDt(mzList, iList, dt, mzStart, mzEnd);
		}


	QList<double> 
		MassSpectrumJsPrototype::mzArray() const
		{
			return thisMassSpectrum()->keyList();
		}


	QList<double> 
		MassSpectrumJsPrototype::iArray() const
		{
			return thisMassSpectrum()->valList();
		}


	QString 
		MassSpectrumJsPrototype::asText()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			double rangeStart = -1;
			double rangeEnd = -1;

			if(argCount != 0 && argCount != 2)
			{
				ctx->throwError("Syntax error: function asText takes either none or two numbers as index arguments.");

				return QString();
			}

			if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function asText() takes zero or two numbers as index arguments.");
					return QString();
				}

				rangeStart = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as index arguments.");
					return QString();
				}

				rangeEnd = qvariant_cast<double>(arg.data().toVariant());
			}

			// Finally, do the work.

			QString *text = thisMassSpectrum()->asText(rangeStart, rangeEnd);

			QString result = *text;
			delete text;

			return result;
		}


	double 
		MassSpectrumJsPrototype::tic() const
		{
			// Either there is no argument, or there must be two arguments:
			// mzRangeStart and mzRangeEnd

			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 0)
				return thisMassSpectrum()->valSum();

			if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function tic() takes two numbers as arguments.");

					return -1;
				}

				double mzRangeStart = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function tic() takes two numbers as arguments.");

					return -1;
				}

				double mzRangeEnd = qvariant_cast<double>(arg.data().toVariant());

				return thisMassSpectrum()->valSum(mzRangeStart, mzRangeEnd);
			}

			ctx->throwError("Syntax error: function tic() takes either no argument "
					"or two numbers as arguments.");

			return -1;
		}


	int 
		MassSpectrumJsPrototype::combine()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{

				// The single argument can be either DataPoint or MassSpectrum

				QScriptValue arg = ctx->argument(0);

				QVariant variant = arg.data().toVariant();

				if(!variant.isValid())
				{
					qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
						<< "Variant is not valid";

					ctx->throwError("Syntax error: function combine takes one DataPoint object "
							"or one MassSpectrum object or two numbers (mz & i) as argument.");

					return -1;
				}

				// Test if DataPoint

				if(variant.userType() == QMetaType::type("msXpSlibmass::DataPoint"))
				{
					DataPoint dp(qscriptvalue_cast<DataPoint>(arg));

					if(!dp.isValid())
					{
						ctx->throwError("Function combine takes one initialized "
								"DataPoint object as argument.");

						return -1;
					}

					return thisMassSpectrum()->combine(dp);
				}

				// Test if MassSpectrum

				if(variant.userType() == QMetaType::type("msXpSlibmass::MassSpectrum"))
				{
					MassSpectrum ms(qscriptvalue_cast<MassSpectrum>(arg));

					return thisMassSpectrum()->combine(ms);
				}	
			}
			else if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as arguments.");
					return -1;
				}

				double mz = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as arguments.");
					return -1;
				}

				double i = qvariant_cast<double>(arg.data().toVariant());

				return thisMassSpectrum()->combine(DataPoint(mz, i));
			}

			ctx->throwError("Syntax error: function combine takes one DataPoint "
					"or one MassSpectrum object or two numbers (mz & i) as argument.");

			return -1;
		}


	int 
		MassSpectrumJsPrototype::subtract()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{

				// The single argument can be either DataPoint or MassSpectrum

				QScriptValue arg = ctx->argument(0);

				QVariant variant = arg.data().toVariant();

				if(!variant.isValid())
				{
					qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
						<< "Variant is not valid";

					ctx->throwError("Syntax error: function subtract takes one DataPoint object "
							"or one MassSpectrum object or two numbers (mz & i) as argument.");

					return -1;
				}

				// Test if DataPoint

				if(variant.userType() == QMetaType::type("msXpSlibmass::DataPoint"))
				{
					DataPoint mp(qscriptvalue_cast<DataPoint>(arg));

					if(!mp.isValid())
					{
						ctx->throwError("Function subtract takes one initialized "
								"DataPoint object as argument.");

						return -1;
					}

					return thisMassSpectrum()->subtract(mp);
				}

				// Test if MassSpectrum

				if(variant.userType() == QMetaType::type("msXpSlibmass::MassSpectrum"))
				{
					MassSpectrum ms(qscriptvalue_cast<MassSpectrum>(arg));

					return thisMassSpectrum()->subtract(ms);
				}	
			}
			else if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function subtract() takes two numbers as arguments.");
					return -1;
				}

				double mz = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function subtract() takes two numbers as arguments.");
					return -1;
				}

				double i = qvariant_cast<double>(arg.data().toVariant());

				return thisMassSpectrum()->subtract(DataPoint(mz, i));
			}

			ctx->throwError("Syntax error: function subtract takes one DataPoint object "
					"or one MassSpectrum object or two numbers (mz & i) as argument.");

			return -1;
		}


	QScriptValue 
		MassSpectrumJsPrototype::valueOf() const
		{
			return thisObject().data();
		}


} // namespace msXpSlibmass
