/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment sphere example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Guillaume Emont <guillaume@fluendo.com>
 */

#include <pgm/pgm.h>
#include <cairo.h>
#include <math.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* Offset added to the sphere */
#define OFFSET 20

/* Take a screenshot of the current viewport content */
static void
take_screenshot (PgmViewport *viewport)
{
  gint width = 0, height = 0;
  guint8 *pixels = NULL;

  pgm_viewport_get_size (viewport, &width, &height);

  /* Pixels retrieved are in the RGBA color space. The pixel buffer shall
   * be cleaned in the 'pixels-read' signal handler. */
  pixels = g_malloc (width * height * 4);

  /* Take a screenshot of the whole viewport */
  pgm_viewport_read_pixels (viewport, 0, 0, width, height, pixels);
}

/* "pixels-read" handler */
static void
pixels_read_cb (PgmViewport *viewport,
                guint width,
                guint height,
                gpointer pixels,
                gpointer data)
{
  GdkPixbuf *pixbuf = NULL;

  pixbuf = gdk_pixbuf_new_from_data (pixels, GDK_COLORSPACE_RGB, TRUE, 8,
                                     width, height, width*4, NULL, NULL);
  gdk_pixbuf_save (pixbuf, "screenshot.png", "png", NULL, NULL);

  g_print ("Screenshot saved in './screenshot.png'\n");

  g_object_unref (pixbuf);
  pixbuf = NULL;
  g_free (pixels);
  pixels = NULL;
}

/* Key-press handler */
static void
key_press_event_cb (PgmViewport *viewport,
                    PgmEventKey *event,
                    PgmDrawable *drawable)
{
  gfloat x, y, z;

  pgm_drawable_get_position (drawable, &x, &y, &z);
  switch (event->keyval)
    {
    case PGM_Up:
      y -= OFFSET;
      break;
    case PGM_Down:
      y += OFFSET;
      break;
    case PGM_Left:
      x -= OFFSET;
      break;
    case PGM_Right:
      x += OFFSET;
      break;
    case PGM_s:
    case PGM_S:
      z -= OFFSET;
      break;
    case PGM_x:
    case PGM_X:
      z += OFFSET;
      break;
   case PGM_p:
   case PGM_P:
      take_screenshot (viewport);
      break;

    case PGM_q:
    case PGM_Q:
      pgm_main_quit ();
      return;

    default:
      return;
    }

  pgm_drawable_set_position (drawable, x, y, z);
}

/* Allocates a memory buffer in data and draws a gradient sphere with
 * the Cairo library in it */
PgmError
cairo_drawing (guchar **data,
               gint width,
               gint height)
{
  PgmError err = PGM_ERROR_OK;
  cairo_pattern_t *pat = NULL;
  cairo_surface_t *surface;
  cairo_t *ctx;

  if (!data)
    return PGM_ERROR_X;

  *data = g_malloc0 (width * height * 4 * sizeof (guchar));
  surface = cairo_image_surface_create_for_data (*data, CAIRO_FORMAT_ARGB32,
                                                 width, height, width * 4);
  ctx = cairo_create (surface);
  if (!(surface && ctx))
    {
      err = PGM_ERROR_X;
      goto tidy;
    }
  cairo_scale (ctx, width, height);

  pat = cairo_pattern_create_radial (0.45, 0.4, 0.1, 0.4, 0.4, 0.5);
  if (!pat)
    {
      err = PGM_ERROR_X;
      goto tidy;
    }
  cairo_pattern_add_color_stop_rgba (pat, 0, 0.8, 0.8, 1, 1);
  cairo_pattern_add_color_stop_rgba (pat, 1, 0, 0, 1, 1);

  cairo_set_source (ctx, pat);
  cairo_arc (ctx, 0.5, 0.5, 0.3, 0, 2 * M_PI);
  cairo_fill (ctx);

tidy:
  cairo_pattern_destroy (pat);
  cairo_surface_destroy (surface);
  cairo_destroy (ctx);

  return err;
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  PgmError ret = PGM_ERROR_OK;
  PgmViewport *viewport = NULL;
  PgmCanvas *canvas;
  PgmDrawable *img;
  guchar *data;

  g_print ("arrows : Move the sphere around\n"
           "'s'    : Zoom out\n"
           "'x'    : Zoom in\n"
           "'p'    : Take a screenshot\n\n");

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_title (viewport, "Cairo sphere");

  g_signal_connect (G_OBJECT (viewport), "pixels-read",
                    G_CALLBACK (pixels_read_cb), NULL);

  /* sphere creation */
  ret = cairo_drawing (&data, 400, 400);
  g_assert (PGM_ERROR_OK == ret);

  img = pgm_image_new_from_buffer (PGM_IMAGE_BGRA, 400, 400, 400 * 4,
                                   400 * 400 * 4, data);
  pgm_drawable_set_bg_color (img, 255, 255, 255, 0);
  pgm_drawable_set_position (img, 200, 100, 0);
  pgm_drawable_set_size (img, 400, 400);
  pgm_drawable_show (img);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 800, 600);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, PGM_DRAWABLE (img));

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (key_press_event_cb), img);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  g_object_unref (canvas);
  g_object_unref (viewport);
  pgm_deinit ();
  g_free (data);

  return 0;
}
