/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of python-ubuntu-platform-api.
 *
 * python-upa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * python-upa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <Python.h>

#include <ubuntu/ui/ubuntu_ui_session_service.h>
#include <ubuntu/application/ui/display.h>


/// These represent python callables we call when each different event happens.
static PyObject* usr_session_requested_cb = NULL;
static PyObject* usr_session_born_cb = NULL;
static PyObject* usr_session_focused_cb = NULL;
static PyObject* usr_session_unfocused_cb = NULL;
static PyObject* usr_session_died_cb = NULL;

// The observer itself:
static ubuntu_ui_session_lifecycle_observer observer;

// forward declared functions:
static PyObject* py_set_session_requested_cb(PyObject* dummy, PyObject* args);
static PyObject* py_set_session_born_cb(PyObject* dummy, PyObject* args);
static PyObject* py_set_session_focused_cb(PyObject* dummy, PyObject* args);
static PyObject* py_set_session_unfocused_cb(PyObject* dummy, PyObject* args);
static PyObject* py_set_session_died_cb(PyObject* dummy, PyObject* args);
static PyObject* set_cb(PyObject *callable, PyObject** cb_to_set);
static PyObject* py_start_monitoring(PyObject*, PyObject*);
static const char* get_well_known_app_name(ubuntu_ui_well_known_application app);
static PyObject* get_arglist_dict_from_session_properties(ubuntu_ui_session_properties props);
static void on_session_requested(ubuntu_ui_well_known_application app, void*);
static void on_session_born(ubuntu_ui_session_properties props, void*);
static void on_session_focused(ubuntu_ui_session_properties props, void*);
static void on_session_unfocused(ubuntu_ui_session_properties props, void*);
static void on_session_died(ubuntu_ui_session_properties props, void*);




static PyObject* py_set_session_requested_cb(PyObject* dummy, PyObject* args)
{
    PyObject* callable;

    if (PyArg_ParseTuple(args, "O:set_session_requested_callback", &callable)) {
        if (!PyCallable_Check(callable)) {
            PyErr_SetString(PyExc_TypeError, "parameter must be callable");
            return NULL;
        }
        return set_cb(callable, &usr_session_requested_cb);
    }
    return NULL;
}

static PyObject* py_set_session_born_cb(PyObject* dummy, PyObject* args)
{
    PyObject* callable;

    if (PyArg_ParseTuple(args, "O:set_session_born_callback", &callable)) {
        if (!PyCallable_Check(callable)) {
            PyErr_SetString(PyExc_TypeError, "parameter must be callable");
            return NULL;
        }
        return set_cb(callable, &usr_session_born_cb);
    }
    return NULL;
}

static PyObject* py_set_session_focused_cb(PyObject* dummy, PyObject* args)
{
    PyObject* callable;

    if (PyArg_ParseTuple(args, "O:set_session_focused_callback", &callable)) {
        if (!PyCallable_Check(callable)) {
            PyErr_SetString(PyExc_TypeError, "parameter must be callable");
            return NULL;
        }
        return set_cb(callable, &usr_session_focused_cb);
    }
    return NULL;
}

static PyObject* py_set_session_unfocused_cb(PyObject* dummy, PyObject* args)
{
    PyObject* callable;

    if (PyArg_ParseTuple(args, "O:set_session_unfocused_callback", &callable)) {
        if (!PyCallable_Check(callable)) {
            PyErr_SetString(PyExc_TypeError, "parameter must be callable");
            return NULL;
        }
        return set_cb(callable, &usr_session_unfocused_cb);
    }
    return NULL;
}

static PyObject* py_set_session_died_cb(PyObject* dummy, PyObject* args)
{
    PyObject* callable;

    if (PyArg_ParseTuple(args, "O:set_session_died_callback", &callable)) {
        if (!PyCallable_Check(callable)) {
            PyErr_SetString(PyExc_TypeError, "parameter must be callable");
            return NULL;
        }
        return set_cb(callable, &usr_session_died_cb);
    }
    return NULL;
}

/// Do the boilderplate code to actually store the callable object:
static PyObject* set_cb(PyObject *callable, PyObject** cb_to_set)
{
    Py_XINCREF(callable);        /* Add a reference to new callback */
    Py_XDECREF(*cb_to_set);      /* Dispose of previous callback */
    *cb_to_set = callable;       /* Remember new callback */

    Py_RETURN_NONE;
}

static PyObject* py_start_monitoring(PyObject*, PyObject*)
{
    memset(&observer, 0, sizeof(observer));
    observer.on_session_requested = on_session_requested;
    observer.on_session_born = on_session_born;
    observer.on_session_unfocused = on_session_unfocused;
    observer.on_session_focused = on_session_focused;
    observer.on_session_died = on_session_died;

    ubuntu_ui_session_install_session_lifecycle_observer(&observer);

    Py_RETURN_NONE;
}

static const char* get_well_known_app_name(ubuntu_ui_well_known_application app)
{
    switch (app)
    {
        case UNKNOWN_APP:
        return "unknown";
        case CAMERA_APP:
        return "camera";
        case GALLERY_APP:
        return "gallery";
        case BROWSER_APP:
        return "browser";
        case SHARE_APP:
        return "share";
        default:
        return "other";
    }
}

/// Get a PyObject* representing a dictionary of all the information we can extract
/// from the session properties object. The return value must be DECREF'd.
static PyObject* get_arglist_dict_from_session_properties(ubuntu_ui_session_properties props)
{
    return Py_BuildValue("({s:i,s:s})",
        "app-instance-id", ubuntu_ui_session_properties_get_application_instance_id(props),
        "desktop-file-hint", ubuntu_ui_session_properties_get_desktop_file_hint(props)
        );
}

static PyObject* py_get_resolution(PyObject* dummy, PyObject* args)
{
    UAUiDisplay* display = ua_ui_display_new_with_index(0); // get info for screen 0.
    int32_t res_x = ua_ui_display_query_horizontal_res(display);
    int32_t res_y = ua_ui_display_query_vertical_res(display);
    ua_ui_display_destroy(display);
    return Py_BuildValue("(i,i)", res_x, res_y);
}

static void on_session_requested(ubuntu_ui_well_known_application app, void*)
{
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();
    if (usr_session_requested_cb)
    {
        PyObject* arglist = Py_BuildValue("({s:s})", "application", get_well_known_app_name(app));
        PyObject* result = PyObject_CallObject(usr_session_requested_cb, arglist);
        if (result)
            Py_DECREF(result);
        else
            PyErr_Print();
        Py_DECREF(arglist);
    }
    PyGILState_Release(gstate);
}

static void on_session_born(ubuntu_ui_session_properties props, void*)
{
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();
    if (usr_session_born_cb)
    {
        PyObject* args = get_arglist_dict_from_session_properties(props);
        PyObject* result = PyObject_CallObject(usr_session_born_cb, args);
        if (result)
            Py_DECREF(result);
        else
            PyErr_Print();
        Py_DECREF(args);
    }
    PyGILState_Release(gstate);
}

static void on_session_focused(ubuntu_ui_session_properties props, void*)
{
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();
    if (usr_session_focused_cb)
    {
        PyObject* args = get_arglist_dict_from_session_properties(props);
        PyObject* result = PyObject_CallObject(usr_session_focused_cb, args);
        if (result)
            Py_DECREF(result);
        else
            PyErr_Print();
        Py_DECREF(args);
    }
    PyGILState_Release(gstate);
}

static void on_session_unfocused(ubuntu_ui_session_properties props, void*)
{
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();
    if (usr_session_unfocused_cb)
    {
        PyObject* args = get_arglist_dict_from_session_properties(props);
        PyObject* result = PyObject_CallObject(usr_session_unfocused_cb, args);
        if (result)
            Py_DECREF(result);
        else
            PyErr_Print();
        Py_DECREF(args);
    }
    PyGILState_Release(gstate);
}

static void on_session_died(ubuntu_ui_session_properties props, void*)
{
    PyGILState_STATE gstate;
    gstate = PyGILState_Ensure();
    if (usr_session_died_cb)
    {
        PyObject* args = get_arglist_dict_from_session_properties(props);
        PyObject* result = PyObject_CallObject(usr_session_died_cb, args);
        if (result)
            Py_DECREF(result);
        else
            PyErr_Print();
        Py_DECREF(args);
    }
    PyGILState_Release(gstate);
}

static PyMethodDef UPAMethods[] = {
    {"set_session_requested_callback", py_set_session_requested_cb, METH_VARARGS, "Set callback for 'session requested'."},
    {"set_session_born_callback", py_set_session_born_cb, METH_VARARGS, "Set callback for 'session born'."},
    {"set_session_focused_callback", py_set_session_focused_cb, METH_VARARGS, "Set callback for 'session focused'."},
    {"set_session_unfocused_callback", py_set_session_unfocused_cb, METH_VARARGS, "Set callback for 'session unfocused'."},
    {"set_session_died_callback", py_set_session_died_cb, METH_VARARGS, "Set callback for 'session died'."},
    {"get_resolution", py_get_resolution, METH_VARARGS, "Get horizontal and verticle resolution."},
    {"start_monitoring", py_start_monitoring, METH_VARARGS, "Start monitoring."},
    {NULL, NULL, 0, NULL}        /* Sentinel */
};


PyMODINIT_FUNC init_sf(void)
{
    PyEval_InitThreads();
    (void) Py_InitModule("_sf", UPAMethods);
}
